import 'dart:convert';

import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/domain/auth_session.dart';

void main() {
  group('AuthRepository', () {
    late InMemoryAuthStorage storage;
    late Uri baseUri;
    late List<String> events;

    setUp(() {
      storage = InMemoryAuthStorage();
      baseUri = Uri.parse('https://natinfo.app/api');
      events = [];
    });

    test('login stores session and decodes payload', () async {
      final handler = MockClient((request) async {
        if (request.url.path == '/api/auth/login/') {
          return http.Response(
            jsonEncode({
              'access': _buildToken(expMinutes: 60, userId: 7),
              'refresh': 'refresh_token',
              'refresh_expires_at':
                  DateTime.now()
                      .toUtc()
                      .add(const Duration(days: 7))
                      .toIso8601String(),
              'has_doc_pro': true,
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        if (request.url.path == '/api/auth/me/') {
          return http.Response(
            jsonEncode({
              'username': 'retio',
              'user_id': 7,
              'has_doc_pro': true,
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        return http.Response('not found', 404);
      });

      final repository = AuthRepository(
        baseUri: baseUri,
        storage: storage,
        httpClient: handler,
      );

      final session = await repository.login(
        username: 'user',
        password: 'pass',
      );

      expect(session.userId, 7);
      expect(session.username, 'retio');
      expect(session.hasDocPro, isTrue);
      expect(await storage.read(), isNotNull);
    });

    test('uses fallback username/user_id when claims are missing', () async {
      final handler = MockClient((request) async {
        if (request.url.path == '/api/auth/login/') {
          return http.Response(
            jsonEncode({
              'access': _buildToken(
                expMinutes: 60,
                userId: 0,
                omitUsername: true,
                omitUserId: true,
              ),
              'refresh': 'refresh_token',
              'user_id': 99,
              'refresh_expires_at':
                  DateTime.now()
                      .toUtc()
                      .add(const Duration(days: 7))
                      .toIso8601String(),
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        if (request.url.path == '/api/auth/me/') {
          return http.Response(
            jsonEncode({
              'username': 'profileUser',
              'user_id': 101,
              'has_doc_pro': true,
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        return http.Response('not found', 404);
      });

      final repository = AuthRepository(
        baseUri: baseUri,
        storage: storage,
        httpClient: handler,
      );

      final session = await repository.login(
        username: 'fallbackUser',
        password: 'pass',
      );

      expect(session.username, 'profileUser');
      expect(session.userId, 101);
      expect(session.hasDocPro, isTrue);
    });

    test('refresh rotates tokens and persists them', () async {
      var refreshCalled = false;
      final handler = MockClient((request) async {
        if (request.url.path == '/api/auth/refresh/') {
          refreshCalled = true;
          return http.Response(
            jsonEncode({
              'access': _buildToken(expMinutes: 30, userId: 7),
              'refresh': 'new_refresh',
              'refresh_expires_at':
                  DateTime.now()
                      .toUtc()
                      .add(const Duration(days: 5))
                      .toIso8601String(),
              'has_doc_pro': true,
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        if (request.url.path == '/api/auth/me/') {
          return http.Response(
            jsonEncode({
              'username': 'profile_after_refresh',
              'user_id': 7,
              'has_doc_pro': true,
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        return http.Response('not found', 404);
      });

      final repository = AuthRepository(
        baseUri: baseUri,
        storage: storage,
        httpClient: handler,
      );

      repository.addSessionListener(() => events.add('refreshed'));

      final initial = AuthSession.fromTokens(
        access: _buildToken(expMinutes: -1, userId: 7),
        refresh: 'refresh_token',
        refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 5)),
      );
      await storage.save(initial);

      await repository.loadSession();
      final refreshed = await repository.refreshToken();

      expect(refreshCalled, isTrue);
      expect(refreshed!.refreshToken, 'new_refresh');
      expect((await storage.read())!.refreshToken, 'new_refresh');
      expect(events.contains('refreshed'), isTrue);
      expect(refreshed.hasDocPro, isTrue);
      expect(refreshed.username, 'profile_after_refresh');
    });

    test('authorized client refreshes expiring token before request', () async {
      final refreshResponse = _buildToken(expMinutes: 30, userId: 1);
      final requests = <String>[];
      final handler = MockClient((request) async {
        requests.add(request.url.path);
        if (request.url.path == '/api/auth/refresh/') {
          return http.Response(
            jsonEncode({
              'access': refreshResponse,
              'refresh': 'refresh_token',
              'refresh_expires_at':
                  DateTime.now()
                      .toUtc()
                      .add(const Duration(days: 7))
                      .toIso8601String(),
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        if (request.url.path == '/api/protected') {
          expect(request.headers['Authorization'], 'Bearer $refreshResponse');
          return http.Response('ok', 200);
        }
        return http.Response('not found', 404);
      });

      final repository = AuthRepository(
        baseUri: baseUri,
        storage: storage,
        httpClient: handler,
        accessLeeway: const Duration(hours: 1),
      );

      final expiring = AuthSession.fromTokens(
        access: _buildToken(expMinutes: 0, userId: 1),
        refresh: 'refresh_token',
        refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
      );
      await storage.save(expiring);

      final client = repository.authorizedClient;
      final response = await client.get(_url(baseUri, 'protected'));

      expect(response.statusCode, 200);
      expect(requests.contains('/api/auth/refresh/'), isTrue);
    });

    test('authorized client retries once on 401 after refresh', () async {
      var firstProtectedCall = true;
      final newAccess = _buildToken(expMinutes: 15, userId: 42);
      final handler = MockClient((request) async {
        if (request.url.path == '/api/protected') {
          if (firstProtectedCall) {
            firstProtectedCall = false;
            return http.Response('unauthorized', 401);
          }
          expect(request.headers['Authorization'], 'Bearer $newAccess');
          return http.Response('ok', 200);
        }
        if (request.url.path == '/api/auth/refresh/') {
          return http.Response(
            jsonEncode({
              'access': newAccess,
              'refresh': 'refresh_token',
              'refresh_expires_at':
                  DateTime.now()
                      .toUtc()
                      .add(const Duration(days: 7))
                      .toIso8601String(),
            }),
            200,
            headers: {'content-type': 'application/json'},
          );
        }
        return http.Response('not found', 404);
      });

      final repository = AuthRepository(
        baseUri: baseUri,
        storage: storage,
        httpClient: handler,
      );

      final session = AuthSession.fromTokens(
        access: _buildToken(expMinutes: 30, userId: 42),
        refresh: 'refresh_token',
        refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
      );
      await storage.save(session);

      final client = repository.authorizedClient;
      final response = await client.get(_url(baseUri, 'protected'));

      expect(response.statusCode, 200);
    });

    test(
      'logout clears storage, notifies listeners, and stops auth headers',
      () async {
        var listenerCalled = false;
        final handler = MockClient((request) async {
          if (request.url.path == '/api/auth/logout/') {
            return http.Response('', 204);
          }
          if (request.url.path == '/api/protected') {
            expect(request.headers.containsKey('Authorization'), isFalse);
            return http.Response('ok', 200);
          }
          return http.Response('not found', 404);
        });

        final repository = AuthRepository(
          baseUri: baseUri,
          storage: storage,
          httpClient: handler,
        );
        final session = AuthSession.fromTokens(
          access: _buildToken(expMinutes: 30, userId: 1),
          refresh: 'refresh_token',
          refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
        );
        await storage.save(session);
        repository.addSessionListener(() {
          listenerCalled = true;
        });

        await repository.logout();

        expect(listenerCalled, isTrue);
        expect(await storage.read(), isNull);

        final response = await repository.authorizedClient.get(
          _url(baseUri, 'protected'),
        );
        expect(response.statusCode, 200);
      },
    );
  });
}

String _buildToken({
  required int expMinutes,
  required int userId,
  bool omitUsername = false,
  bool omitUserId = false,
}) {
  final header = base64UrlEncode(
    utf8.encode(jsonEncode({'alg': 'HS256', 'typ': 'JWT'})),
  );
  final payload = base64UrlEncode(
    utf8.encode(
      jsonEncode({
        'exp':
            DateTime.now()
                .toUtc()
                .add(Duration(minutes: expMinutes))
                .millisecondsSinceEpoch ~/
            1000,
        if (!omitUserId) 'user_id': userId,
        if (!omitUsername) 'username': 'user$userId',
      }),
    ),
  );
  const signature = 'signature';
  return '$header.$payload.$signature';
}

Uri _url(Uri base, String path) {
  final normalizedBase =
      base.toString().endsWith('/') ? base.toString() : '${base.toString()}/';
  return Uri.parse('$normalizedBase$path');
}
