import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';

enum UnitInput { mgAir, gSang }

enum DriverType { all, normal, probatoire, eleve, accompagnateur }

enum VehicleType { all, normal, transportCommun }

enum InfractionMode { mesure, ivresseManifeste, refusVerifications }

enum Domain { route, rail, eauxInterieures, maritime, teleoperateur }

enum EauxType { general, passagers, marchandisesDangereuses }

enum TeleOpTransport { personnes, marchandises }

enum MaritimeRole { all, pilote, fonctionsABord }

class CalculateurTauxAlcoolemiePage extends StatefulWidget {
  const CalculateurTauxAlcoolemiePage({super.key});
  @override
  State<CalculateurTauxAlcoolemiePage> createState() =>
      _CalculateurTauxAlcoolemiePageState();
}

class _CalculateurTauxAlcoolemiePageState
    extends State<CalculateurTauxAlcoolemiePage> {
  final TextEditingController _controller = TextEditingController();

  // Options communes
  UnitInput _unit = UnitInput.mgAir;
  InfractionMode _mode = InfractionMode.mesure;
  Domain _domain = Domain.route;

  // Route
  DriverType _driver = DriverType.normal;
  VehicleType _vehicle = VehicleType.normal;
  bool _ead = false;
  bool _eadNeutraliseAutocar = false;
  bool _aideNeutralisation = false;
  bool _violationsDeliberes = false;

  // Contexte global
  bool _recidive = false;
  bool _stup = false; // stupéfiants concomitants

  // Eaux intérieures
  EauxType _eauxType = EauxType.general;

  // Intervenant à distance
  TeleOpTransport _teleOp = TeleOpTransport.personnes;

  // Maritime
  MaritimeRole _maritimeRole = MaritimeRole.pilote;

  // Résultats
  double? _retenu;
  String? _classification;
  String? _explication;
  List<String>? _natinfCodes;
  List<Natinf>? _natinfResults;

  // UI
  String? _errorText;
  bool _isInputValid = false;
  bool _isLoading = false;

  @override
  void initState() {
    super.initState();
    MatomoService().trackPage(
      title: "Calculateur de taux d'alcoolémie",
      path: '/tools/alcoolemie/',
    );
    _controller.addListener(() {
      final text = _controller.text.replaceAll(',', '.');
      setState(() {
        _isInputValid = text.isNotEmpty && double.tryParse(text) != null;
      });
    });
  }

  // ---------- conversions & portées ----------
  // g/L sang -> mg/L air  (0,5 g/L ≈ 0,25 mg/L)
  double _toMgAir(double value, UnitInput unit) =>
      unit == UnitInput.mgAir ? value : value * 0.5;

  bool _coversDriver(DriverType want) {
    if (_driver == DriverType.all) return true;
    if (want == DriverType.normal) {
      return _driver == DriverType.normal ||
          _driver == DriverType.accompagnateur;
    }
    return _driver == want;
  }

  bool _coversVehicle(VehicleType want) =>
      _vehicle == VehicleType.all ? true : _vehicle == want;

  // ---------- mapping ROUTE ----------
  List<String> _routeContrav(double retenu) {
    final codes = <String>{};
    final qualifiesSpecial = retenu >= 0.10;
    final qualifiesNormal = retenu >= 0.25;

    if (_ead && qualifiesSpecial) codes.add('33329'); // conducteur soumis EAD
    if (_coversVehicle(VehicleType.transportCommun) && qualifiesSpecial) {
      codes.add('25434'); // transport en commun
    }
    if (_coversDriver(DriverType.probatoire) && qualifiesSpecial)
      codes.add('31060');
    if (_coversDriver(DriverType.eleve) && qualifiesSpecial) codes.add('31061');
    if (_coversDriver(DriverType.normal) &&
        _coversVehicle(VehicleType.normal) &&
        qualifiesNormal) {
      codes.add('13322'); // normal / accompagnateur
    }
    // Autocar : neutralisation EAD
    if (_eadNeutraliseAutocar) codes.add('28672');
    if (_aideNeutralisation) codes.add('28673');

    return codes.toList();
  }

  // Délit route (retenu ≥ 0,40 mg/L)
  List<String> _routeDelit() {
    if (_stup) return [_recidive ? '29259' : '23762']; // stup + alcool
    if (_violationsDeliberes) return ['32809']; // violations délibérées
    return [_recidive ? '8544' : '1247']; // alcool seul
  }

  // ---------- mapping RAIL ----------
  // Contrav ≥ 0,10 mg/L
  List<String> _railCodes(double retenu) =>
      retenu >= 0.10 ? ['31564'] : <String>[];

  // ---------- mapping EAUX INTÉRIEURES ----------
  // Contraventions :
  //  - 35735 : passagers ≥ 0,10 mg/L
  //  - 35733 : général ≥ 0,25 mg/L
  List<String> _eauxContravCodes(double retenu) {
    final codes = <String>[];
    if (_eauxType == EauxType.passagers && retenu >= 0.10) codes.add('35735');
    if (_eauxType != EauxType.passagers && retenu >= 0.25) codes.add('35733');
    return codes;
  }

  // Délits eaux intérieures :
  //  - 34092 : stup + alcool
  //  - 34080 / 34081 / 2279 selon type (tous à ≥ 0,40 mg/L)
  List<String> _eauxDelitCodes(double retenu) {
    if (_stup) return ['34092'];
    if (retenu >= 0.40) {
      switch (_eauxType) {
        case EauxType.passagers:
          return ['34080'];
        case EauxType.marchandisesDangereuses:
          return ['34081'];
        case EauxType.general:
        default:
          return ['2279'];
      }
    }
    return <String>[];
  }

  // ---------- mapping MARITIME ----------
  // Délits ≥ 0,25 mg/L
  List<String> _maritimeCodes(double retenu) {
    if (retenu < 0.25) return <String>[];
    if (_maritimeRole == MaritimeRole.all) return ['28018', '32193'];
    return [_maritimeRole == MaritimeRole.pilote ? '28018' : '32193'];
  }

  // Refus maritime (pilote / fonctions à bord)
  List<String> _maritimeRefus() {
    if (_maritimeRole == MaritimeRole.all) return ['32198', '32199'];
    return [_maritimeRole == MaritimeRole.pilote ? '32198' : '32199'];
  }

  // ---------- mapping TÉLÉOPÉRATEUR ----------
  List<String> _teleOpCodes(double retenu) {
    if (retenu >= 0.40) {
      return [_stup ? '34806' : '34802']; // délits
    }
    if (_teleOp == TeleOpTransport.personnes && retenu >= 0.10)
      return ['34808'];
    if (_teleOp == TeleOpTransport.marchandises && retenu >= 0.25)
      return ['34809'];
    return <String>[];
  }

  // ---------- refus & ivresse ----------
  List<String> _refusCodes() {
    switch (_domain) {
      case Domain.teleoperateur:
        return ['34804'];
      case Domain.maritime:
        return _maritimeRefus();
      case Domain.route:
        return _violationsDeliberes ? ['32810'] : [_recidive ? '9164' : '51'];
      default:
        return <String>[]; // non modélisé pour les autres domaines
    }
  }

  List<String> _ivresseCodes() => [_recidive ? '9009' : '41'];

  // ---------- calcul ----------
  void _calculate() async {
    // Modes sans taux
    if (_mode == InfractionMode.ivresseManifeste) {
      final codes = _ivresseCodes();
      setState(() {
        _retenu = null;
        _classification = 'Délictuel';
        _explication = "Infraction : ivresse manifeste (sans éthylomètre).";
        _natinfCodes = codes;
        _natinfResults = null;
      });
      _fetchNatinfDetails(codes);
      return;
    }
    if (_mode == InfractionMode.refusVerifications) {
      final codes = _refusCodes();
      setState(() {
        _retenu = null;
        _classification = 'Délictuel';
        _explication = "Infraction : refus de se soumettre aux vérifications.";
        _natinfCodes = codes;
        _natinfResults = null;
      });
      _fetchNatinfDetails(codes);
      return;
    }

    // Mode "mesure"
    final text = _controller.text.replaceAll(',', '.');
    final rawInput = double.tryParse(text);
    if (rawInput == null) {
      setState(() {
        _errorText = 'Veuillez entrer un nombre valide';
        _clearResults(keepOptions: true);
      });
      return;
    }

    final inputMgAir = _toMgAir(rawInput, _unit);

    double raw;
    String exp;
    if (_unit == UnitInput.gSang) {
      raw = inputMgAir; // pas de déduction pour mesure sanguine
      exp =
          "${rawInput.toStringAsFixed(2)} g/L × 0.5 = ${raw.toStringAsFixed(4)} mg/L → arrondi à ";
    } else {
      // Éthylomètre : déductions
      if (inputMgAir < 0.4) {
        raw = inputMgAir - 0.032;
        exp =
            "${inputMgAir.toStringAsFixed(2)} - 0.032 = ${raw.toStringAsFixed(4)} → arrondi à ";
      } else if (inputMgAir <= 2.0) {
        raw = inputMgAir - (inputMgAir * 0.08);
        exp =
            "${inputMgAir.toStringAsFixed(2)} - (${inputMgAir.toStringAsFixed(2)} × 0.08) = ${raw.toStringAsFixed(4)} → arrondi à ";
      } else {
        raw = inputMgAir - (inputMgAir * 0.30);
        exp =
            "${inputMgAir.toStringAsFixed(2)} - (${inputMgAir.toStringAsFixed(2)} × 0.30) = ${raw.toStringAsFixed(4)} → arrondi à ";
      }
    }

    final retenu = (raw * 100).floorToDouble() / 100;

    // Codes par domaine
    List<String> codes = <String>[];
    String classif = 'Aucune infraction';

    switch (_domain) {
      case Domain.route:
        if (retenu >= 0.40) {
          codes = _routeDelit();
          classif = 'Délictuel';
        } else {
          codes = _routeContrav(retenu);
          classif =
              codes.isNotEmpty
                  ? (_driver == DriverType.all || _vehicle == VehicleType.all
                      ? 'Contraventionnel (selon type)'
                      : 'Contraventionnel')
                  : 'Aucune infraction';
        }
        break;

      case Domain.rail:
        codes = _railCodes(retenu);
        classif = codes.isNotEmpty ? 'Contraventionnel' : 'Aucune infraction';
        break;

      case Domain.eauxInterieures:
        final delits = _eauxDelitCodes(retenu);
        if (delits.isNotEmpty) {
          codes = delits;
          classif = 'Délictuel';
        } else {
          final contrav = _eauxContravCodes(retenu);
          codes = contrav;
          classif =
              contrav.isNotEmpty ? 'Contraventionnel' : 'Aucune infraction';
        }
        break;

      case Domain.maritime:
        codes = _maritimeCodes(retenu); // délits (≥ 0,25 mg/L)
        classif = codes.isNotEmpty ? 'Délictuel' : 'Aucune infraction';
        break;

      case Domain.teleoperateur:
        codes = _teleOpCodes(retenu);
        classif =
            codes.isEmpty
                ? 'Aucune infraction'
                : (retenu >= 0.40 ? 'Délictuel' : 'Contraventionnel');
        break;
    }

    setState(() {
      _errorText = null;
      _retenu = retenu;
      _classification = classif;
      _explication = exp + retenu.toStringAsFixed(2) + ' mg/L (air)';
      _natinfCodes = codes;
      _natinfResults = null;
    });

    if (codes.isNotEmpty) {
      _fetchNatinfDetails(codes);
    }
  }

  Future<void> _fetchNatinfDetails(List<String> codes) async {
    setState(() {
      _isLoading = true;
      _natinfResults = null;
    });
    final results = <Natinf>[];
    for (final code in codes.toSet()) {
      final natinf = await DatabaseHelper().getNatinfByNumero(code);
      if (natinf != null) results.add(natinf);
    }
    if (!mounted) return;
    setState(() {
      _isLoading = false;
      _natinfResults = results;
    });
  }

  void _clearResults({bool keepOptions = false}) {
    _retenu = null;
    _classification = null;
    _explication = null;
    _natinfCodes = null;
    _natinfResults = null;
    _isLoading = false;
    if (!keepOptions) {
      _unit = UnitInput.mgAir;
      _mode = InfractionMode.mesure;
      _domain = Domain.route;
      _driver = DriverType.normal;
      _vehicle = VehicleType.normal;
      _ead = false;
      _eadNeutraliseAutocar = false;
      _aideNeutralisation = false;
      _violationsDeliberes = false;
      _recidive = false;
      _stup = false;
      _eauxType = EauxType.general;
      _teleOp = TeleOpTransport.personnes;
      _maritimeRole = MaritimeRole.pilote;
    }
  }

  Future<void> _launchArrete() async {
    final uri = Uri.parse(
      'https://www.legifrance.gouv.fr/loda/id/JORFTEXT000000420869/',
    );
    if (!await launchUrl(uri)) {
      if (!mounted) return;
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Impossible d\'ouvrir le lien')),
      );
    }
  }

  @override
  void dispose() {
    _controller.dispose();
    super.dispose();
  }

  // ---------- UI helpers ----------
  DropdownButtonFormField<T> _buildDropdown<T>({
    required T value,
    required List<DropdownMenuItem<T>> items,
    required void Function(T?) onChanged,
    String? label,
  }) {
    return DropdownButtonFormField<T>(
      value: value,
      items: items,
      onChanged: onChanged,
      isExpanded: true,
      decoration: InputDecoration(
        labelText: label,
        border: const OutlineInputBorder(),
        contentPadding: const EdgeInsets.symmetric(
          horizontal: 12,
          vertical: 10,
        ),
      ),
    );
  }

  SwitchListTile _buildSwitch({
    required bool value,
    required void Function(bool) onChanged,
    required String title,
    String? subtitle,
  }) {
    return SwitchListTile(
      value: value,
      onChanged: onChanged,
      title: Text(title),
      subtitle: subtitle != null ? Text(subtitle) : null,
      contentPadding: EdgeInsets.zero,
    );
  }

  Widget _buildDisclaimerCard() {
    final cs = Theme.of(context).colorScheme;
    return Container(
      decoration: BoxDecoration(
        color: cs.surfaceVariant,
        borderRadius: BorderRadius.circular(10),
      ),
      padding: const EdgeInsets.all(12),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: const [
          Icon(Icons.info_outline),
          SizedBox(width: 8),
          Expanded(
            child: Text(
              "Ce calculateur fournit une orientation basée sur des seuils usuels "
              "et un sous-ensemble de NATINF. Selon les circonstances, d’autres "
              "infractions ou aggravations peuvent s’appliquer et prévaloir. "
              "Vérifiez toujours la qualification retenue et les textes en vigueur.",
              style: TextStyle(height: 1.25),
            ),
          ),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final unitItems = <DropdownMenuItem<UnitInput>>[
      const DropdownMenuItem(
        value: UnitInput.mgAir,
        child: Text("mg/L d'air expiré"),
      ),
      const DropdownMenuItem(
        value: UnitInput.gSang,
        child: Text("g/L de sang"),
      ),
    ];

    final modeItems = <DropdownMenuItem<InfractionMode>>[
      const DropdownMenuItem(
        value: InfractionMode.mesure,
        child: Text('Mesure alcoolémie'),
      ),
      const DropdownMenuItem(
        value: InfractionMode.ivresseManifeste,
        child: Text('Ivresse manifeste (sans éthylotest)'),
      ),
      const DropdownMenuItem(
        value: InfractionMode.refusVerifications,
        child: Text('Refus des vérifications'),
      ),
    ];

    final domainItems = <DropdownMenuItem<Domain>>[
      const DropdownMenuItem(value: Domain.route, child: Text('Route')),
      const DropdownMenuItem(value: Domain.rail, child: Text('Rail')),
      const DropdownMenuItem(
        value: Domain.eauxInterieures,
        child: Text('Eaux intérieures'),
      ),
      const DropdownMenuItem(value: Domain.maritime, child: Text('Maritime')),
      const DropdownMenuItem(
        value: Domain.teleoperateur,
        child: Text('Intervenant à distance'),
      ),
    ];

    final driverItems = <DropdownMenuItem<DriverType>>[
      const DropdownMenuItem(
        value: DriverType.all,
        child: Text('Tous les types'),
      ),
      const DropdownMenuItem(
        value: DriverType.normal,
        child: Text('Conducteur normal / accompagnateur'),
      ),
      const DropdownMenuItem(
        value: DriverType.probatoire,
        child: Text('Permis probatoire'),
      ),
      const DropdownMenuItem(
        value: DriverType.eleve,
        child: Text('Élève conducteur'),
      ),
      const DropdownMenuItem(
        value: DriverType.accompagnateur,
        child: Text('Accompagnateur'),
      ),
    ];

    final vehicleItems = <DropdownMenuItem<VehicleType>>[
      const DropdownMenuItem(
        value: VehicleType.all,
        child: Text('Tous les véhicules'),
      ),
      const DropdownMenuItem(
        value: VehicleType.normal,
        child: Text('Véhicule normal'),
      ),
      const DropdownMenuItem(
        value: VehicleType.transportCommun,
        child: Text('Transport en commun'),
      ),
    ];

    final eauxItems = <DropdownMenuItem<EauxType>>[
      const DropdownMenuItem(value: EauxType.general, child: Text('Général')),
      const DropdownMenuItem(
        value: EauxType.passagers,
        child: Text('Passagers'),
      ),
      const DropdownMenuItem(
        value: EauxType.marchandisesDangereuses,
        child: Text('Marchandises dangereuses'),
      ),
    ];

    final teleOpItems = <DropdownMenuItem<TeleOpTransport>>[
      const DropdownMenuItem(
        value: TeleOpTransport.personnes,
        child: Text('Transport de personnes'),
      ),
      const DropdownMenuItem(
        value: TeleOpTransport.marchandises,
        child: Text('Transport de marchandises'),
      ),
    ];

    final maritimeRoleItems = <DropdownMenuItem<MaritimeRole>>[
      const DropdownMenuItem(
        value: MaritimeRole.all,
        child: Text('Tous les rôles'),
      ),
      const DropdownMenuItem(
        value: MaritimeRole.pilote,
        child: Text('Pilote de navire'),
      ),
      const DropdownMenuItem(
        value: MaritimeRole.fonctionsABord,
        child: Text('Fonctions à bord'),
      ),
    ];

    final isMeasure = _mode == InfractionMode.mesure;

    return Scaffold(
      appBar: AppBar(title: const Text("Calculateur de taux d'alcoolémie")),
      body: LayoutBuilder(
        builder: (context, constraints) {
          final isWide = constraints.maxWidth >= 640;

          final tauxField = TextField(
            controller: _controller,
            enabled: isMeasure,
            keyboardType: const TextInputType.numberWithOptions(decimal: true),
            textInputAction: TextInputAction.done,
            onSubmitted:
                (_) => (isMeasure && _isInputValid) ? _calculate() : null,
            decoration: InputDecoration(
              labelText:
                  _unit == UnitInput.mgAir
                      ? "Taux affiché sur l'éthylomètre (mg/L air)"
                      : "Taux sanguin (g/L de sang)",
              hintText: _unit == UnitInput.mgAir ? 'ex : 0.43' : 'ex : 0.86',
              helperText:
                  _unit == UnitInput.mgAir
                      ? "Valeur en mg/L d'air expiré"
                      : "Valeur en g/L de sang",
              errorText: _errorText,
              border: const OutlineInputBorder(),
            ),
          );

          return SingleChildScrollView(
            padding: const EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.stretch,
              children: [
                // Mode + Domaine
                if (isWide)
                  Row(
                    children: [
                      Expanded(
                        child: _buildDropdown<InfractionMode>(
                          value: _mode,
                          items: modeItems,
                          label: 'Mode d’infraction',
                          onChanged:
                              (v) => setState(() {
                                _mode = v!;
                                _clearResults(keepOptions: true);
                              }),
                        ),
                      ),
                      const SizedBox(width: 12),
                      Expanded(
                        child: _buildDropdown<Domain>(
                          value: _domain,
                          items: domainItems,
                          label: 'Domaine',
                          onChanged:
                              (v) => setState(() {
                                _domain = v!;
                                _clearResults(keepOptions: true);
                              }),
                        ),
                      ),
                    ],
                  )
                else
                  Column(
                    children: [
                      _buildDropdown<InfractionMode>(
                        value: _mode,
                        items: modeItems,
                        label: 'Mode d’infraction',
                        onChanged:
                            (v) => setState(() {
                              _mode = v!;
                              _clearResults(keepOptions: true);
                            }),
                      ),
                      const SizedBox(height: 12),
                      _buildDropdown<Domain>(
                        value: _domain,
                        items: domainItems,
                        label: 'Domaine',
                        onChanged:
                            (v) => setState(() {
                              _domain = v!;
                              _clearResults(keepOptions: true);
                            }),
                      ),
                    ],
                  ),

                const SizedBox(height: 16),

                // Taux + Unité (si Mesure)
                if (isMeasure)
                  (isWide
                      ? Row(
                        children: [
                          Expanded(flex: 2, child: tauxField),
                          const SizedBox(width: 12),
                          Expanded(
                            child: _buildDropdown<UnitInput>(
                              value: _unit,
                              items: unitItems,
                              label: 'Unité de saisie',
                              onChanged:
                                  (v) => setState(() {
                                    _unit = v!;
                                    _clearResults(keepOptions: true);
                                  }),
                            ),
                          ),
                        ],
                      )
                      : Column(
                        children: [
                          tauxField,
                          const SizedBox(height: 12),
                          _buildDropdown<UnitInput>(
                            value: _unit,
                            items: unitItems,
                            label: 'Unité de saisie',
                            onChanged:
                                (v) => setState(() {
                                  _unit = v!;
                                  _clearResults(keepOptions: true);
                                }),
                          ),
                        ],
                      )),

                const SizedBox(height: 16),

                // Options spécifiques par domaine
                if (_domain == Domain.route) ...[
                  if (isWide)
                    Row(
                      children: [
                        Expanded(
                          child: _buildDropdown<DriverType>(
                            value: _driver,
                            items: driverItems,
                            label: 'Type de conducteur',
                            onChanged:
                                (v) => setState(() {
                                  _driver = v!;
                                  _clearResults(keepOptions: true);
                                }),
                          ),
                        ),
                        const SizedBox(width: 12),
                        Expanded(
                          child: _buildDropdown<VehicleType>(
                            value: _vehicle,
                            items: vehicleItems,
                            label: 'Type de véhicule',
                            onChanged:
                                (v) => setState(() {
                                  _vehicle = v!;
                                  _clearResults(keepOptions: true);
                                }),
                          ),
                        ),
                      ],
                    )
                  else ...[
                    _buildDropdown<DriverType>(
                      value: _driver,
                      items: driverItems,
                      label: 'Type de conducteur',
                      onChanged:
                          (v) => setState(() {
                            _driver = v!;
                            _clearResults(keepOptions: true);
                          }),
                    ),
                    const SizedBox(height: 12),
                    _buildDropdown<VehicleType>(
                      value: _vehicle,
                      items: vehicleItems,
                      label: 'Type de véhicule',
                      onChanged:
                          (v) => setState(() {
                            _vehicle = v!;
                            _clearResults(keepOptions: true);
                          }),
                    ),
                  ],
                  _buildSwitch(
                    value: _ead,
                    onChanged: (v) {
                      setState(() {
                        _ead = v;
                        _clearResults(keepOptions: true);
                      });
                    },
                    title: "Éthylotest anti-démarrage (EAD)",
                  ),
                  _buildSwitch(
                    value: _eadNeutraliseAutocar,
                    onChanged: (v) {
                      setState(() {
                        _eadNeutraliseAutocar = v;
                        if (!v) _aideNeutralisation = false;
                        _clearResults(keepOptions: true);
                      });
                    },
                    title: "EAD neutralisé (autocar)",
                  ),
                  if (_eadNeutraliseAutocar)
                    _buildSwitch(
                      value: _aideNeutralisation,
                      onChanged: (v) {
                        setState(() {
                          _aideNeutralisation = v;
                          _clearResults(keepOptions: true);
                        });
                      },
                      title: "Aide à la neutralisation",
                    ),
                  _buildSwitch(
                    value: _violationsDeliberes,
                    onChanged: (v) {
                      setState(() {
                        _violationsDeliberes = v;
                        _clearResults(keepOptions: true);
                      });
                    },
                    title:
                        "Violations délibérées de la réglementation routière",
                    subtitle:
                        "Conduite compromettant la sécurité / tranquillité publiques",
                  ),
                ] else if (_domain == Domain.eauxInterieures) ...[
                  _buildDropdown<EauxType>(
                    value: _eauxType,
                    items: eauxItems,
                    label: 'Type de bateau',
                    onChanged:
                        (v) => setState(() {
                          _eauxType = v!;
                          _clearResults(keepOptions: true);
                        }),
                  ),
                ] else if (_domain == Domain.teleoperateur) ...[
                  _buildDropdown<TeleOpTransport>(
                    value: _teleOp,
                    items: teleOpItems,
                    label: 'Type de transport',
                    onChanged:
                        (v) => setState(() {
                          _teleOp = v!;
                          _clearResults(keepOptions: true);
                        }),
                  ),
                ] else if (_domain == Domain.maritime) ...[
                  _buildDropdown<MaritimeRole>(
                    value: _maritimeRole,
                    items: maritimeRoleItems,
                    label: 'Rôle maritime',
                    onChanged:
                        (v) => setState(() {
                          _maritimeRole = v!;
                          _clearResults(keepOptions: true);
                        }),
                  ),
                ],

                // Contexte global
                const SizedBox(height: 8),
                _buildSwitch(
                  value: _stup,
                  onChanged: (v) {
                    setState(() {
                      _stup = v;
                      _clearResults(keepOptions: true);
                    });
                  },
                  title: "Stupéfiants concomitants",
                ),
                _buildSwitch(
                  value: _recidive,
                  onChanged: (v) {
                    setState(() {
                      _recidive = v;
                      _clearResults(keepOptions: true);
                    });
                  },
                  title: "Récidive",
                ),

                const SizedBox(height: 12),
                ElevatedButton.icon(
                  onPressed:
                      (_mode == InfractionMode.mesure ? _isInputValid : true)
                          ? _calculate
                          : null,
                  icon: const Icon(Icons.calculate),
                  label: const Text('Calculer'),
                ),

                const SizedBox(height: 24),

                // Résultats
                if (_mode == InfractionMode.mesure && _retenu != null) ...[
                  Text(
                    'Taux retenu : ${_retenu!.toStringAsFixed(2)} mg/L',
                    style: const TextStyle(
                      fontSize: 18,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  const SizedBox(height: 8),
                ],
                if (_explication != null)
                  Text(
                    'Calcul effectué : $_explication',
                    style: const TextStyle(fontStyle: FontStyle.italic),
                  ),
                const SizedBox(height: 8),
                if (_classification != null)
                  Text('Classification : $_classification'),

                const SizedBox(height: 16),
                if (_isLoading)
                  const Center(child: CircularProgressIndicator())
                else if (_natinfResults != null &&
                    _natinfResults!.isNotEmpty) ...[
                  Text(
                    _natinfResults!.length > 1
                        ? 'Code(s) NATINF applicable(s) :'
                        : 'Code NATINF applicable :',
                    style: const TextStyle(
                      fontSize: 16,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  const SizedBox(height: 8),
                  NatinfList(
                    natinfList: _natinfResults!,
                    recordHistory: false,
                    nested: true,
                  ),
                ] else if (_natinfCodes != null &&
                    _natinfCodes!.isNotEmpty) ...[
                  Text('Code(s) NATINF : ${_natinfCodes!.join(', ')}'),
                ],

                if ((_natinfCodes != null && _natinfCodes!.isNotEmpty) ||
                    (_natinfResults != null && _natinfResults!.isNotEmpty)) ...[
                  const SizedBox(height: 12),
                  _buildDisclaimerCard(),
                ],

                const SizedBox(height: 8),
                const Divider(height: 32),
              ],
            ),
          );
        },
      ),
      bottomNavigationBar:
          (_mode == InfractionMode.mesure && _retenu != null)
              ? Padding(
                padding: const EdgeInsets.all(16.0),
                child: TextButton(
                  onPressed: _launchArrete,
                  child: const Text(
                    'Arrêté du 8 juillet 2003 relatif au contrôle des éthylomètres – Légifrance',
                    textAlign: TextAlign.center,
                    style: TextStyle(decoration: TextDecoration.underline),
                  ),
                ),
              )
              : null,
    );
  }
}
