import 'package:flutter/material.dart';
import 'package:flutter_colorpicker/flutter_colorpicker.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/features/downloads/presentation/download_manager_page.dart';
import 'package:natinfo_flutter/app/state/operational_mode_provider.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/settings/data/app_icon_service.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';

class _SettingsSection {
  const _SettingsSection({
    required this.id,
    required this.title,
    required this.subtitle,
    required this.icon,
    required this.buildContent,
  });

  final String id;
  final String title;
  final String subtitle;
  final IconData icon;
  final List<Widget> Function(BuildContext context) buildContent;
}

/// Settings screen regrouping app options by theme.
class SettingsPage extends StatefulWidget {
  /// Builds the settings page.
  const SettingsPage({super.key});

  @override
  State<SettingsPage> createState() => _SettingsPageState();
}

class _SettingsPageState extends State<SettingsPage> {
  final AppIconService _appIconService = AppIconService();
  final Map<String, GlobalKey> _sectionKeys = {};
  bool _forceOffline = AppConfig.forceOffline;
  bool _matomoTrackingEnabled = true;
  bool _iconChangeSupported = false;
  bool _isLoadingAppIcon = true;
  String _selectedIconKey = AppIconService.defaultKey;

  @override
  void initState() {
    super.initState();
    _loadOfflineSetting();
    _loadTrackingSetting();
    _initAppIconState();
    MatomoService().trackPage(title: 'Réglages', path: '/settings/');
  }

  Future<void> _initAppIconState() async {
    try {
      final supported = await _appIconService.isSupported();
      if (!mounted) {
        return;
      }
      if (!supported) {
        setState(() {
          _iconChangeSupported = false;
          _isLoadingAppIcon = false;
        });
        return;
      }

      final currentKey = await _appIconService.getCurrentIconKey();
      if (!mounted) {
        return;
      }
      setState(() {
        _iconChangeSupported = true;
        _selectedIconKey = currentKey;
        _isLoadingAppIcon = false;
      });
    } catch (error) {
      if (!mounted) {
        return;
      }
      setState(() {
        _iconChangeSupported = false;
        _isLoadingAppIcon = false;
      });
    }
  }

  Future<void> _loadOfflineSetting() async {
    final prefs = await SharedPreferences.getInstance();
    final offline = prefs.getBool('forceOffline') ?? false;
    setState(() {
      _forceOffline = offline;
      AppConfig.forceOffline = offline;
    });
  }

  Future<void> _saveOfflineSetting(bool value) async {
    final prefs = await SharedPreferences.getInstance();
    await prefs.setBool('forceOffline', value);
  }

  Future<void> _loadTrackingSetting() async {
    final prefs = await SharedPreferences.getInstance();
    final enabled = prefs.getBool('matomoTrackingEnabled') ?? true;
    setState(() {
      _matomoTrackingEnabled = enabled;
    });
  }

  Future<void> _saveTrackingSetting(bool value) async {
    final prefs = await SharedPreferences.getInstance();
    await prefs.setBool('matomoTrackingEnabled', value);
  }

  Future<void> _saveOperationalMode(bool value) async {
    final prefs = await SharedPreferences.getInstance();
    await prefs.setBool('operationalMode', value);
  }

  GlobalKey _sectionKey(String id) {
    return _sectionKeys.putIfAbsent(id, () => GlobalKey());
  }

  Future<void> _scrollToSection(String id) async {
    final key = _sectionKeys[id];
    final targetContext = key?.currentContext;
    if (targetContext == null) {
      return;
    }
    await Scrollable.ensureVisible(
      targetContext,
      duration: const Duration(milliseconds: 240),
      alignment: 0.08,
      curve: Curves.easeInOut,
    );
  }

  Widget _buildQuickNav(List<_SettingsSection> sections) {
    final chips =
        sections
            .map(
              (section) => ActionChip(
                label: Text(section.title),
                avatar: Icon(section.icon, size: 18),
                onPressed: () => _scrollToSection(section.id),
              ),
            )
            .toList();

    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text('Accès rapide', style: Theme.of(context).textTheme.titleMedium),
        const SizedBox(height: 8),
        SingleChildScrollView(
          scrollDirection: Axis.horizontal,
          child: Row(
            children: [
              for (var i = 0; i < chips.length; i++) ...[
                chips[i],
                if (i != chips.length - 1) const SizedBox(width: 8),
              ],
            ],
          ),
        ),
      ],
    );
  }

  Widget _buildColorPickerTile({
    required String label,
    required Color currentColor,
    required Future<void> Function(Color color) onColorSaved,
    required String dialogTitle,
  }) {
    return ListTile(
      leading: CircleAvatar(backgroundColor: currentColor),
      title: Text(label),
      onTap: () async {
        Color pickerColor = currentColor;
        await showDialog<void>(
          context: context,
          builder: (dialogContext) {
            final navigator = Navigator.of(dialogContext);
            return AlertDialog(
              title: Text(dialogTitle),
              content: SingleChildScrollView(
                child: ColorPicker(
                  pickerColor: pickerColor,
                  onColorChanged: (color) => pickerColor = color,
                  labelTypes: const [ColorLabelType.rgb],
                  pickerAreaHeightPercent: 0.8,
                ),
              ),
              actions: [
                TextButton(
                  onPressed: () => Navigator.of(dialogContext).pop(),
                  child: const Text("Annuler"),
                ),
                TextButton(
                  onPressed: () async {
                    await onColorSaved(pickerColor);
                    navigator.pop();
                  },
                  child: const Text("Valider"),
                ),
              ],
            );
          },
        );
      },
    );
  }

  Future<void> _openIconSelector() async {
    final currentSelection = _selectedIconKey;
    final selectedKey = await showModalBottomSheet<String>(
      context: context,
      isScrollControlled: true,
      builder: (modalContext) {
        return FractionallySizedBox(
          heightFactor: 0.85,
          child: SafeArea(
            child: Padding(
              padding: const EdgeInsets.symmetric(
                horizontal: 16.0,
                vertical: 12.0,
              ),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Row(
                    children: [
                      Expanded(
                        child: Text(
                          "Choisir l'icône",
                          style: Theme.of(modalContext).textTheme.titleLarge,
                        ),
                      ),
                      IconButton(
                        icon: const Icon(Icons.close),
                        onPressed: () => Navigator.of(modalContext).pop(),
                      ),
                    ],
                  ),
                  const SizedBox(height: 12),
                  Expanded(
                    child: GridView.builder(
                      gridDelegate: SliverGridDelegateWithFixedCrossAxisCount(
                        crossAxisCount:
                            MediaQuery.of(modalContext).size.width > 600
                                ? 4
                                : 3,
                        mainAxisSpacing: 16,
                        crossAxisSpacing: 16,
                        childAspectRatio: 0.8,
                      ),
                      itemCount: AppIconService.options.length,
                      itemBuilder: (context, index) {
                        final option = AppIconService.options[index];
                        final isSelected = option.key == currentSelection;
                        return _AppIconOptionTile(
                          option: option,
                          isSelected: isSelected,
                          onTap:
                              () => Navigator.of(modalContext).pop(option.key),
                        );
                      },
                    ),
                  ),
                  Align(
                    alignment: Alignment.centerRight,
                    child: TextButton(
                      onPressed: () => Navigator.of(modalContext).pop(),
                      child: const Text("Fermer"),
                    ),
                  ),
                ],
              ),
            ),
          ),
        );
      },
    );

    if (!mounted || selectedKey == null || selectedKey == _selectedIconKey) {
      return;
    }

    setState(() {
      _isLoadingAppIcon = true;
    });

    try {
      await _appIconService.setIcon(selectedKey);
      if (!mounted) {
        return;
      }
      setState(() {
        _selectedIconKey = selectedKey;
      });
      if (!mounted) {
        return;
      }
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(
          content: Text("L'icône de l’application a été mise à jour."),
        ),
      );
    } catch (error) {
      if (!mounted) {
        return;
      }
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text("Impossible de changer l’icône : $error")),
      );
    } finally {
      if (mounted) {
        setState(() {
          _isLoadingAppIcon = false;
        });
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    final themeProvider = Provider.of<ThemeProvider>(context);
    final selectedIconOption = AppIconService.getOptionByKey(_selectedIconKey);
    final sections = <_SettingsSection>[
      _SettingsSection(
        id: 'connectivity',
        title: 'Connectivité',
        subtitle: 'Recherche locale et mode opérationnel',
        icon: Icons.wifi_off,
        buildContent:
            (context) => [
              SwitchListTile(
                title: const Text('Mode hors ligne'),
                subtitle: const Text('Force les recherches en base locale.'),
                value: _forceOffline,
                onChanged: (value) {
                  setState(() {
                    _forceOffline = value;
                    AppConfig.forceOffline = value;
                  });
                  _saveOfflineSetting(value);
                },
              ),
              Consumer<OperationalModeProvider>(
                builder: (context, opProvider, child) {
                  return SwitchListTile(
                    title: const Text('Mode opérationnel'),
                    subtitle: const Text(
                      'Portrait verrouillé et interface allégée.',
                    ),
                    value: opProvider.opMode,
                    onChanged: (value) {
                      opProvider.opMode = value;
                      _saveOperationalMode(value);
                    },
                  );
                },
              ),
            ],
      ),
      _SettingsSection(
        id: 'appearance',
        title: 'Apparence',
        subtitle: 'Thème, icône et couleur principale',
        icon: Icons.palette_outlined,
        buildContent:
            (context) => [
              SwitchListTile(
                title: const Text('Mode sombre'),
                subtitle: const Text('Active ou désactive le thème sombre.'),
                value: themeProvider.isDarkMode,
                onChanged: themeProvider.toggleTheme,
              ),
              ListTile(
                leading:
                    _isLoadingAppIcon
                        ? const SizedBox(
                          width: 40,
                          height: 40,
                          child: CircularProgressIndicator(strokeWidth: 2.5),
                        )
                        : ClipRRect(
                          borderRadius: BorderRadius.circular(12),
                          child: Image.asset(
                            selectedIconOption.previewAssetPath,
                            width: 40,
                            height: 40,
                            fit: BoxFit.cover,
                          ),
                        ),
                title: const Text("Icône de l’application"),
                subtitle: Text(
                  _iconChangeSupported
                      ? "Actuelle : ${selectedIconOption.label}"
                      : "Personnalisation indisponible sur cet appareil.",
                ),
                enabled: _iconChangeSupported && !_isLoadingAppIcon,
                onTap: _iconChangeSupported ? _openIconSelector : null,
              ),
              const SizedBox(height: 8),
              const Padding(
                padding: EdgeInsets.symmetric(horizontal: 16.0),
                child: Text('Couleur principale'),
              ),
              _buildColorPickerTile(
                label: 'Couleur principale',
                currentColor: themeProvider.primaryColor,
                onColorSaved: themeProvider.updatePrimaryColor,
                dialogTitle: 'Choisir une couleur',
              ),
              Align(
                alignment: Alignment.centerRight,
                child: TextButton(
                  onPressed:
                      () => themeProvider.updatePrimaryColor(Colors.indigo),
                  child: const Text('Restaurer par défaut'),
                ),
              ),
            ],
      ),
      _SettingsSection(
        id: 'infractions',
        title: 'Couleurs des infractions',
        subtitle: 'Codes couleur par type',
        icon: Icons.color_lens,
        buildContent:
            (context) => [
              _buildColorPickerTile(
                label: 'Obligatoire',
                currentColor: themeProvider.obligatoryColor,
                onColorSaved: themeProvider.updateObligatoryColor,
                dialogTitle: "Couleur pour 'Obligatoire'",
              ),
              _buildColorPickerTile(
                label: 'Facultatif',
                currentColor: themeProvider.facultativeColor,
                onColorSaved: themeProvider.updateFacultativeColor,
                dialogTitle: "Couleur pour 'Facultatif'",
              ),
              _buildColorPickerTile(
                label: 'Non',
                currentColor: themeProvider.nonColor,
                onColorSaved: themeProvider.updateNonColor,
                dialogTitle: "Couleur pour 'Non'",
              ),
              _buildColorPickerTile(
                label: 'Crime',
                currentColor: themeProvider.crimeColor,
                onColorSaved: themeProvider.updateCrimeColor,
                dialogTitle: "Couleur pour 'Crime'",
              ),
              _buildColorPickerTile(
                label: 'Délit',
                currentColor: themeProvider.delitColor,
                onColorSaved: themeProvider.updateDelitColor,
                dialogTitle: "Couleur pour 'Délit'",
              ),
              _buildColorPickerTile(
                label: 'Contravention',
                currentColor: themeProvider.contraventionColor,
                onColorSaved: themeProvider.updateContraventionColor,
                dialogTitle: "Couleur pour 'Contravention'",
              ),
              Align(
                alignment: Alignment.centerRight,
                child: TextButton(
                  onPressed: themeProvider.resetInfractionColors,
                  child: const Text('Restaurer les couleurs par défaut'),
                ),
              ),
            ],
      ),
      _SettingsSection(
        id: 'downloads',
        title: 'Données téléchargées',
        subtitle: 'Gestion des données locales',
        icon: Icons.cloud_download_outlined,
        buildContent:
            (context) => [
              ListTile(
                leading: const Icon(Icons.manage_search),
                title: const Text('Ouvrir le gestionnaire'),
                subtitle: const Text(
                  'Consulter et gérer les données téléchargées.',
                ),
                onTap: () {
                  Navigator.push(
                    context,
                    MaterialPageRoute<void>(
                      builder: (context) => const DownloadManagerPage(),
                    ),
                  );
                },
              ),
            ],
      ),
      _SettingsSection(
        id: 'privacy',
        title: 'Confidentialité',
        subtitle: 'Statistiques et suivi',
        icon: Icons.privacy_tip_outlined,
        buildContent:
            (context) => [
              SwitchListTile(
                title: const Text('Suivi statistique (Matomo)'),
                subtitle: const Text('Active les mesures d’usage anonymisées.'),
                value: _matomoTrackingEnabled,
                onChanged: (value) {
                  setState(() {
                    _matomoTrackingEnabled = value;
                  });
                  _saveTrackingSetting(value);
                },
              ),
            ],
      ),
    ];

    return Scaffold(
      appBar: AppBar(title: const Text("Réglages")),
      body: SingleChildScrollView(
        child: Padding(
          padding: const EdgeInsets.all(16.0),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              _buildQuickNav(sections),
              const SizedBox(height: 16),
              ...sections.map((section) {
                final key = _sectionKey(section.id);
                return Card(
                  key: key,
                  margin: const EdgeInsets.only(bottom: 16),
                  shape: RoundedRectangleBorder(
                    borderRadius: BorderRadius.circular(12),
                  ),
                  child: ExpansionTile(
                    initiallyExpanded: true,
                    tilePadding: const EdgeInsets.symmetric(horizontal: 16),
                    leading: Icon(section.icon),
                    title: Text(section.title),
                    subtitle: Text(
                      section.subtitle,
                      style: Theme.of(context).textTheme.bodySmall,
                    ),
                    childrenPadding: const EdgeInsets.only(
                      left: 8,
                      right: 8,
                      bottom: 12,
                    ),
                    children: section.buildContent(context),
                  ),
                );
              }),
            ],
          ),
        ),
      ),
    );
  }
}

class _AppIconOptionTile extends StatelessWidget {
  const _AppIconOptionTile({
    required this.option,
    required this.isSelected,
    required this.onTap,
  });

  final AppIconOption option;
  final bool isSelected;
  final VoidCallback onTap;

  @override
  Widget build(BuildContext context) {
    final borderColor =
        isSelected
            ? Theme.of(context).colorScheme.primary
            : Theme.of(context).colorScheme.outlineVariant;

    return InkWell(
      onTap: onTap,
      borderRadius: BorderRadius.circular(16),
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          Container(
            decoration: BoxDecoration(
              borderRadius: BorderRadius.circular(16),
              border: Border.all(color: borderColor, width: isSelected ? 3 : 1),
            ),
            padding: const EdgeInsets.all(6),
            child: Stack(
              children: [
                ClipRRect(
                  borderRadius: BorderRadius.circular(12),
                  child: Image.asset(
                    option.previewAssetPath,
                    width: 72,
                    height: 72,
                    fit: BoxFit.cover,
                  ),
                ),
                if (isSelected)
                  Positioned(
                    top: 6,
                    right: 6,
                    child: CircleAvatar(
                      radius: 12,
                      backgroundColor: Theme.of(context).colorScheme.primary,
                      child: const Icon(
                        Icons.check,
                        size: 16,
                        color: Colors.white,
                      ),
                    ),
                  ),
              ],
            ),
          ),
          const SizedBox(height: 8),
          Text(
            option.label,
            textAlign: TextAlign.center,
            style: TextStyle(
              fontWeight: isSelected ? FontWeight.w600 : FontWeight.w400,
            ),
          ),
          if (option.description != null) ...[
            const SizedBox(height: 4),
            Text(
              option.description!,
              textAlign: TextAlign.center,
              style: Theme.of(context).textTheme.bodySmall,
            ),
          ],
        ],
      ),
    );
  }
}
