/// Représente un dossier de favoris stocké localement.
class FavouriteFolder {
  const FavouriteFolder({
    required this.id,
    required this.name,
    this.parentId,
    required this.createdAt,
    this.uuid,
    this.updatedAt,
    this.position = 0,
    this.deleted = false,
  });

  /// Identifiant unique du dossier.
  final int id;

  /// Identifiant stable (UUID) pour la synchronisation.
  final String? uuid;

  /// Nom affiché pour le dossier.
  final String name;

  /// Identifiant du dossier parent, ou `null` pour la racine.
  final int? parentId;

  /// Date de création du dossier.
  final DateTime createdAt;

  /// Date de dernière mise à jour.
  final DateTime? updatedAt;

  /// Position du dossier par rapport à ses frères.
  final int position;

  /// Indique si le dossier est marqué supprimé (tombstone).
  final bool deleted;

  /// Crée une instance à partir d'une carte [map].
  factory FavouriteFolder.fromMap({
    required int id,
    required Map<String, dynamic> map,
  }) {
    return FavouriteFolder(
      id: id,
      uuid: map['uuid'] as String? ?? map['id'] as String?,
      name: map['name'] as String? ?? 'Dossier',
      parentId: map['parent_id'] as int?,
      createdAt: DateTime.fromMillisecondsSinceEpoch(
        map['created_at'] as int? ?? 0,
      ),
      updatedAt: _parseDate(map['updated_at']) ?? _parseDate(map['updatedAt']),
      position: map['position'] is int ? map['position'] as int : 0,
      deleted: map['deleted'] == true,
    );
  }

  /// Convertit le dossier en carte pour l'enregistrement en base.
  Map<String, dynamic> toMap() => {
    'uuid': uuid,
    'name': name,
    'parent_id': parentId,
    'created_at': createdAt.millisecondsSinceEpoch,
    'updated_at': (updatedAt ?? createdAt).millisecondsSinceEpoch,
    'position': position,
    'deleted': deleted,
  };

  /// Renvoie une copie modifiée du dossier.
  FavouriteFolder copyWith({
    String? name,
    int? parentId,
    String? uuid,
    DateTime? updatedAt,
    int? position,
    bool? deleted,
  }) {
    return FavouriteFolder(
      id: id,
      uuid: uuid ?? this.uuid,
      name: name ?? this.name,
      parentId: parentId ?? this.parentId,
      createdAt: createdAt,
      updatedAt: updatedAt ?? this.updatedAt,
      position: position ?? this.position,
      deleted: deleted ?? this.deleted,
    );
  }
}

DateTime? _parseDate(dynamic value) {
  if (value is int) {
    return DateTime.fromMillisecondsSinceEpoch(value);
  }
  if (value is String && value.isNotEmpty) {
    return DateTime.tryParse(value);
  }
  return null;
}
