import 'dart:convert';

/// Holds the authentication state derived from NATINFo JWT tokens.
class AuthSession {
  const AuthSession({
    required this.accessToken,
    required this.refreshToken,
    required this.accessExpiresAt,
    required this.refreshExpiresAt,
    required this.userId,
    required this.username,
    this.tokenLabel,
    this.hasDocPro,
    this.jti,
    this.lastRefresh,
  });

  /// Access token to be sent in the `Authorization` header.
  final String accessToken;

  /// Refresh token used to rotate the access token.
  final String refreshToken;

  /// Expiration timestamp of the current access token.
  final DateTime accessExpiresAt;

  /// Expiration timestamp of the refresh token (rotation window).
  final DateTime refreshExpiresAt;

  /// User identifier from the JWT payload.
  final int? userId;

  /// Username associated with the session.
  final String? username;

  /// Optional token label returned by the backend.
  final String? tokenLabel;

  /// Flag indicating DocPro access when exposed by the payload.
  final bool? hasDocPro;

  /// Token identifier when provided by the backend.
  final String? jti;

  /// Timestamp of the last refresh.
  final DateTime? lastRefresh;

  /// Builds a session from raw tokens and optional metadata.
  factory AuthSession.fromTokens({
    required String access,
    required String refresh,
    required DateTime refreshExpiresAt,
    String? tokenLabel,
    String? jti,
    DateTime? issuedAt,
    String? fallbackUsername,
    int? fallbackUserId,
    bool? fallbackHasDocPro,
  }) {
    final payload = _decodeJwtPayload(access);
    final exp = payload['exp'] is num ? payload['exp'] as num : null;
    final accessExp =
        exp != null
            ? DateTime.fromMillisecondsSinceEpoch(
              exp.toInt() * 1000,
              isUtc: true,
            )
            : DateTime.now().toUtc();
    return AuthSession(
      accessToken: access,
      refreshToken: refresh,
      accessExpiresAt: accessExp,
      refreshExpiresAt: refreshExpiresAt.toUtc(),
      userId: payload['user_id'] as int? ?? fallbackUserId,
      username:
          payload['username'] as String? ??
          payload['user'] as String? ??
          fallbackUsername,
      tokenLabel: tokenLabel,
      hasDocPro: payload['has_doc_pro'] as bool? ?? fallbackHasDocPro,
      jti: jti ?? payload['jti'] as String?,
      lastRefresh: issuedAt,
    );
  }

  /// Returns true if the access token is already expired.
  bool get isAccessExpired => DateTime.now().toUtc().isAfter(accessExpiresAt);

  /// Returns true if the refresh token is expired.
  bool get isRefreshExpired => DateTime.now().toUtc().isAfter(refreshExpiresAt);

  /// Returns true when the access token will expire within the provided window.
  bool willAccessExpireWithin(Duration window) =>
      accessExpiresAt.difference(DateTime.now().toUtc()) <= window;

  /// Serializes the session for persistence.
  Map<String, dynamic> toJson() => {
    'access': accessToken,
    'refresh': refreshToken,
    'access_expires_at': accessExpiresAt.toIso8601String(),
    'refresh_expires_at': refreshExpiresAt.toIso8601String(),
    'user_id': userId,
    'username': username,
    'token_label': tokenLabel,
    'has_doc_pro': hasDocPro,
    'jti': jti,
    'last_refresh': lastRefresh?.toIso8601String(),
  };

  /// Restores a session from persisted data.
  factory AuthSession.fromJson(Map<String, dynamic> json) {
    return AuthSession(
      accessToken: json['access'] as String,
      refreshToken: json['refresh'] as String,
      accessExpiresAt:
          DateTime.parse(json['access_expires_at'] as String).toUtc(),
      refreshExpiresAt:
          DateTime.parse(json['refresh_expires_at'] as String).toUtc(),
      userId: json['user_id'] as int?,
      username: json['username'] as String?,
      tokenLabel: json['token_label'] as String?,
      hasDocPro: json['has_doc_pro'] as bool?,
      jti: json['jti'] as String?,
      lastRefresh:
          json['last_refresh'] != null
              ? DateTime.parse(json['last_refresh'] as String).toUtc()
              : null,
    );
  }

  AuthSession copyWith({
    String? accessToken,
    String? refreshToken,
    DateTime? accessExpiresAt,
    DateTime? refreshExpiresAt,
    int? userId,
    String? username,
    String? tokenLabel,
    bool? hasDocPro,
    String? jti,
    DateTime? lastRefresh,
  }) {
    return AuthSession(
      accessToken: accessToken ?? this.accessToken,
      refreshToken: refreshToken ?? this.refreshToken,
      accessExpiresAt: accessExpiresAt ?? this.accessExpiresAt,
      refreshExpiresAt: refreshExpiresAt ?? this.refreshExpiresAt,
      userId: userId ?? this.userId,
      username: username ?? this.username,
      tokenLabel: tokenLabel ?? this.tokenLabel,
      hasDocPro: hasDocPro ?? this.hasDocPro,
      jti: jti ?? this.jti,
      lastRefresh: lastRefresh ?? this.lastRefresh,
    );
  }

  /// Decodes the payload of a JWT access token.
  static Map<String, dynamic> decodePayload(String token) {
    final parts = token.split('.');
    if (parts.length != 3) {
      return {};
    }
    final normalized = base64Url.normalize(parts[1]);
    final payload = utf8.decode(base64Url.decode(normalized));
    final decoded = jsonDecode(payload);
    return decoded is Map<String, dynamic> ? decoded : {};
  }

  static Map<String, dynamic> _decodeJwtPayload(String token) =>
      decodePayload(token);
}
