import 'dart:convert';

import 'package:flutter_secure_storage/flutter_secure_storage.dart';
import 'package:shared_preferences/shared_preferences.dart';

import '../domain/auth_session.dart';

/// Contract for persisting authentication sessions.
abstract class AuthStorage {
  /// Persists the provided [session].
  Future<void> save(AuthSession session);

  /// Restores a session if available.
  Future<AuthSession?> read();

  /// Clears any stored session.
  Future<void> clear();
}

/// Raised when secure storage operations cannot be completed safely.
class AuthStorageException implements Exception {
  AuthStorageException(this.message, [this.cause]);

  final String message;
  final Object? cause;

  @override
  String toString() => 'AuthStorageException: $message';
}

/// In-memory storage useful for tests.
class InMemoryAuthStorage implements AuthStorage {
  Map<String, dynamic>? _cache;

  @override
  Future<void> clear() async {
    _cache = null;
  }

  @override
  Future<AuthSession?> read() async =>
      _cache != null ? AuthSession.fromJson(_cache!) : null;

  @override
  Future<void> save(AuthSession session) async {
    _cache = session.toJson();
  }
}

/// Storage backed by [FlutterSecureStorage].
class SecureAuthStorage implements AuthStorage {
  SecureAuthStorage({
    FlutterSecureStorage? secureStorage,
    SharedPreferences? sharedPreferences,
  }) : _secureStorage = secureStorage ?? const FlutterSecureStorage(),
       _prefs = sharedPreferences;

  static const _key = 'auth_session';
  final FlutterSecureStorage _secureStorage;
  final SharedPreferences? _prefs;

  @override
  Future<void> clear() async {
    try {
      await _secureStorage.delete(key: _key);
    } catch (err) {
      throw AuthStorageException('Failed to clear secure session', err);
    }
    // Clean up any legacy insecure copy if present.
    await _prefs?.remove(_key);
  }

  @override
  Future<AuthSession?> read() async {
    try {
      final raw = await _secureStorage.read(key: _key);
      if (raw == null) {
        return null;
      }
      final data = jsonDecode(raw);
      return data is Map<String, dynamic> ? AuthSession.fromJson(data) : null;
    } catch (err) {
      throw AuthStorageException(
        'Unable to read session from secure storage',
        err,
      );
    }
  }

  @override
  Future<void> save(AuthSession session) async {
    final serialized = jsonEncode(session.toJson());
    try {
      await _secureStorage.write(key: _key, value: serialized);
    } catch (err) {
      throw AuthStorageException('Unable to persist session securely', err);
    }
    // Ensure legacy insecure copy is purged.
    await _prefs?.remove(_key);
  }
}
