import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

void main() {
  group('SourceRegistry', () {
    const specJson = '''
    {
      "sources": [
        {
          "id": "network-primary",
          "dataset": "natinf",
          "name": "Infractions NATINFo",
          "license": "Licence Ouverte 2.0",
          "type": "api",
          "uri": "https://example.com",
          "scope": "natinf",
          "priority": 2,
          "requiredAtBuild": false,
          "requiresNetwork": true,
          "schemaVersion": 1,
          "checksum": null,
          "checksumAlgo": null
        },
        {
          "id": "local-fallback",
          "dataset": "natinf",
          "name": "Infractions NATINFo",
          "license": "Licence Ouverte 2.0",
          "type": "asset",
          "uri": "asset://natinf.json",
          "scope": "natinf",
          "priority": 1,
          "requiredAtBuild": false,
          "requiresNetwork": false,
          "schemaVersion": 1,
          "checksum": null,
          "checksumAlgo": null
        }
      ]
    }
''';

    test('sorts sources by priority then id', () {
      final registry = SourceRegistry.fromJsonString(specJson);
      final ordered = registry.sourcesForDataset('natinf');
      expect(ordered.map((s) => s.id), ['local-fallback', 'network-primary']);
    });

    test('filters out network sources when disabled', () {
      final registry = SourceRegistry.fromJsonString(specJson);
      final offline = registry.sourcesForDataset('natinf', allowNetwork: false);
      expect(offline, hasLength(1));
      expect(offline.single.type, SourceType.asset);
    });

    test('exposes dataset names', () {
      final registry = SourceRegistry.fromJsonString(specJson);
      expect(registry.datasetNames['natinf'], 'Infractions NATINFo');
      expect(registry.nameForDataset('natinf'), 'Infractions NATINFo');
    });

    test('propagates license to custom overrides', () {
      final registry = SourceRegistry.fromJsonString(
        specJson,
        overrides: {'natinf': Uri.parse('https://override.test')},
      );
      expect(
        registry.sourcesForDataset('natinf').first.license,
        'Licence Ouverte 2.0',
      );
    });

    test('injects custom overrides before defaults', () {
      final registry = SourceRegistry.fromJsonString(
        specJson,
        overrides: {'natinf': Uri.parse('https://override.test')},
      );
      final ordered = registry.sourcesForDataset('natinf');
      expect(ordered.first.id, 'custom-natinf');
      expect(ordered.first.uri.toString(), 'https://override.test');
      expect(ordered.first.requiresNetwork, isTrue);
    });

    test('ignores overrides when source is not user editable', () {
      const lockedSpec = '''
      { "sources": [
        {
          "id": "locked",
          "dataset": "natinf",
          "name": "Infractions NATINFo",
          "license": "Licence Ouverte 2.0",
          "type": "api",
          "uri": "https://locked",
          "scope": "natinf",
          "priority": 1,
          "requiredAtBuild": false,
          "requiresNetwork": true,
          "schemaVersion": 1,
          "checksum": null,
          "checksumAlgo": null,
          "userEditable": false
        }
      ]}
      ''';
      final registry = SourceRegistry.fromJsonString(
        lockedSpec,
        overrides: {'natinf': Uri.parse('https://ignored')},
      );
      final ordered = registry.sourcesForDataset('natinf');
      expect(ordered, hasLength(1));
      expect(ordered.single.id, 'locked');
      expect(ordered.single.uri.toString(), 'https://locked');
    });

    test('throws when dataset names conflict', () {
      expect(
        () => SourceRegistry([
          SourceSpec(
            id: 'first',
            dataset: 'natinf',
            name: 'First name',
            type: SourceType.api,
            uri: Uri.parse('https://example.com'),
            scope: 'natinf',
            priority: 1,
            requiredAtBuild: false,
            requiresNetwork: true,
            schemaVersion: 1,
          ),
          SourceSpec(
            id: 'second',
            dataset: 'natinf',
            name: 'Second name',
            type: SourceType.asset,
            uri: Uri.parse('asset://local.json'),
            scope: 'natinf',
            priority: 2,
            requiredAtBuild: false,
            requiresNetwork: false,
            schemaVersion: 1,
          ),
        ]),
        throwsArgumentError,
      );
    });

    test('throws when registry root is not an object', () {
      const invalidJson = '[]';

      expect(
        () => SourceRegistry.fromJsonString(invalidJson),
        throwsFormatException,
      );
    });
  });
}
