import 'dart:io';

import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/features/pr_locator/data/csv_cache_service.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:natinfo_flutter/shared/data_sources/source_preferences.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';
import 'package:natinfo_flutter/shared/data_sources/update_service.dart';
import 'package:shared_preferences/shared_preferences.dart';

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  group('CsvCacheService', () {
    late Directory tempDir;
    late SharedPreferences prefs;

    setUp(() async {
      tempDir = await Directory.systemTemp.createTemp('csv-cache-test');
      SharedPreferences.setMockInitialValues({});
      prefs = await SharedPreferences.getInstance();
    });

    tearDown(() async {
      await tempDir.delete(recursive: true);
    });

    test(
      'downloads dataset through update service and caches result',
      () async {
        final loader = _FakeLoader();
        final updateService = SourceUpdateService(
          loader: loader,
          cacheDirectoryBuilder: () async => tempDir,
        );
        final csvCache = CsvCacheService(
          updateService: updateService,
          sourcePreferences: SourcePreferences(
            prefs,
            globalNetworkAllowed: true,
          ),
        );

        expect(await csvCache.hasDataset('dataset-test'), isFalse);

        final file = await csvCache.fetchDataset('dataset-test');
        expect(await file.exists(), isTrue);
        expect(await file.readAsString(), equals('content'));
        expect(loader.loadCount, equals(1));

        // Second call should reuse cached file without re-downloading.
        final cached = await csvCache.fetchDataset('dataset-test');
        expect(await cached.readAsString(), equals('content'));
        expect(loader.loadCount, equals(1));
        expect(await csvCache.hasDataset('dataset-test'), isTrue);
      },
    );

    test('detects cached dataset across service instances', () async {
      final loader = _FakeLoader();
      final updateService = SourceUpdateService(
        loader: loader,
        cacheDirectoryBuilder: () async => tempDir,
      );
      final csvCache = CsvCacheService(
        updateService: updateService,
        sourcePreferences: SourcePreferences(prefs, globalNetworkAllowed: true),
      );

      await csvCache.fetchDataset('dataset-cross');
      expect(await csvCache.hasDataset('dataset-cross'), isTrue);

      final secondService = SourceUpdateService(
        loader: _FailingLoader(),
        cacheDirectoryBuilder: () async => tempDir,
        preferences: prefs,
      );
      final secondCache = CsvCacheService(
        updateService: secondService,
        sourcePreferences: SourcePreferences(prefs, globalNetworkAllowed: true),
      );

      expect(await secondCache.hasDataset('dataset-cross'), isTrue);
    });

    test('falls back to payload when metadata cleared', () async {
      final loader = _FakeLoader();
      final updateService = SourceUpdateService(
        loader: loader,
        cacheDirectoryBuilder: () async => tempDir,
      );
      final csvCache = CsvCacheService(
        updateService: updateService,
        sourcePreferences: SourcePreferences(prefs, globalNetworkAllowed: true),
      );

      final file = await csvCache.fetchDataset('dataset-payload-only');
      expect(await file.exists(), isTrue);

      await prefs.remove('sources.dataset-payload-only.lastUpdate');
      await prefs.remove('sources.dataset-payload-only.schemaVersion');
      await prefs.remove('sources.dataset-payload-only.sourceId');

      expect(await csvCache.hasDataset('dataset-payload-only'), isTrue);
    });

    test('relies on metadata when payload file is missing', () async {
      final loader = _FakeLoader();
      final updateService = SourceUpdateService(
        loader: loader,
        cacheDirectoryBuilder: () async => tempDir,
      );
      final csvCache = CsvCacheService(
        updateService: updateService,
        sourcePreferences: SourcePreferences(prefs, globalNetworkAllowed: true),
      );

      final file = await csvCache.fetchDataset('dataset-meta');
      expect(await file.exists(), isTrue);
      await file.delete();

      expect(await csvCache.hasDataset('dataset-meta'), isTrue);
    });

    test('logs dataset detection steps', () async {
      final loader = _FakeLoader();
      final updateService = SourceUpdateService(
        loader: loader,
        cacheDirectoryBuilder: () async => tempDir,
      );
      final recorded = <String>[];
      final csvCache = CsvCacheService(
        updateService: updateService,
        sourcePreferences: SourcePreferences(prefs, globalNetworkAllowed: true),
        logger: SourceLogger(
          tag: 'CsvCacheTest',
          handler: (message) => recorded.add(message),
        ),
      );

      expect(await csvCache.hasDataset('dataset-logs'), isFalse);
      expect(recorded.last, contains('dataset-logs not cached locally'));

      await csvCache.fetchDataset('dataset-logs');
      recorded.clear();

      expect(await csvCache.hasDataset('dataset-logs'), isTrue);
      expect(recorded.last, contains('dataset-logs metadata found'));
    });
  });
}

class _FakeLoader implements SourceLoadClient {
  int loadCount = 0;

  @override
  Future<SourceLoadResult> load(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
    int? expectedSchemaVersion,
  }) async {
    loadCount += 1;
    final spec = SourceSpec(
      id: 'fake-$dataset',
      dataset: dataset,
      name: 'Dataset $dataset',
      type: SourceType.file,
      uri: Uri.parse('file:///tmp/$dataset'),
      scope: dataset,
      priority: 1,
      requiredAtBuild: false,
      requiresNetwork: false,
      schemaVersion: 1,
    );
    final bytes = 'content'.codeUnits;
    return SourceLoadResult(
      spec: spec,
      bytes: bytes,
      integrity: const SourceIntegrity(status: ChecksumStatus.notProvided),
    );
  }
}

class _FailingLoader implements SourceLoadClient {
  @override
  Future<SourceLoadResult> load(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
    int? expectedSchemaVersion,
  }) async {
    fail('Unexpected network load for dataset $dataset');
  }
}
