import 'dart:convert';
import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/domain/auth_session.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_details_page.dart';
import 'package:path_provider_platform_interface/path_provider_platform_interface.dart';
import 'package:provider/provider.dart';

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();
  late bool originalForceOffline;

  setUpAll(() async {
    PathProviderPlatform.instance = _FakePathProviderPlatform();
    await DatabaseHelper().initDb();
  });

  setUp(() {
    originalForceOffline = AppConfig.forceOffline;
    AppConfig.forceOffline = true;
  });

  tearDown(() {
    AppConfig.forceOffline = originalForceOffline;
  });

  testWidgets(
    'shows login CTA when DocPro is unauthorized and user is signed out',
    (tester) async {
      final authProvider = await _buildAuthProvider();
      final repository = _StubNatinfRepository(unauthorized: true);
      const natinf = Natinf(
        numeroNatinf: '12345',
        qualificationInfraction: 'Test',
        dacgSource: 0,
      );

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider(create: (_) => ThemeProvider()),
            ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
            Provider<NatinfRepository>.value(value: repository),
          ],
          child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
        ),
      );

      await tester.pumpAndSettle();

      expect(
        find.textContaining('documents professionnels DocPro'),
        findsOneWidget,
      );
      expect(find.text('Connexion'), findsOneWidget);
    },
  );

  testWidgets(
    'shows DocPro restriction message instead of login CTA when authenticated',
    (tester) async {
      final session = AuthSession.fromTokens(
        access: _token(expMinutes: 60, userId: 42, username: 'alice'),
        refresh: 'refresh',
        refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
      );
      final authProvider = await _buildAuthProvider(session: session);
      final repository = _StubNatinfRepository(unauthorized: true);
      const natinf = Natinf(
        numeroNatinf: '12345',
        qualificationInfraction: 'Test',
        dacgSource: 0,
      );

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider(create: (_) => ThemeProvider()),
            ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
            Provider<NatinfRepository>.value(value: repository),
          ],
          child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
        ),
      );

      await tester.pumpAndSettle();

      expect(
        find.byWidgetPredicate(
          (widget) =>
              widget is RichText &&
              widget.text.toPlainText().contains(
                "votre compte n'y a pas accès",
              ),
        ),
        findsOneWidget,
      );
      expect(find.text('Connexion'), findsNothing);
    },
  );

  testWidgets('shows DocPro documents when available and authorized', (
    tester,
  ) async {
    final session = AuthSession.fromTokens(
      access: _token(expMinutes: 60, userId: 1, username: 'retio'),
      refresh: 'refresh',
      refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
    ).copyWith(hasDocPro: true);
    final authProvider = await _buildAuthProvider(session: session);
    final repository = _StubNatinfRepository(
      unauthorized: false,
      available: true,
      docs: const [
        NatinfDocPro(
          id: 1,
          natinf: '12345',
          title: 'Procédure DocPro',
          url: 'https://example.com/docpro',
        ),
      ],
    );
    const natinf = Natinf(
      numeroNatinf: '12345',
      qualificationInfraction: 'Test',
      dacgSource: 0,
    );

    await tester.pumpWidget(
      MultiProvider(
        providers: [
          ChangeNotifierProvider(create: (_) => ThemeProvider()),
          ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
          Provider<NatinfRepository>.value(value: repository),
        ],
        child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
      ),
    );

    await tester.pumpAndSettle();

    expect(find.text('Procédure DocPro'), findsOneWidget);
    expect(find.text('Connexion'), findsNothing);
  });

  testWidgets(
    'Retrait points à zéro est caché dans la section Plus d’informations',
    (tester) async {
      final session = AuthSession.fromTokens(
        access: _token(expMinutes: 60, userId: 1, username: 'retio'),
        refresh: 'refresh',
        refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
      );
      final authProvider = await _buildAuthProvider(session: session);
      final repository = _StubNatinfRepository(
        unauthorized: false,
        available: false,
        docs: const [],
      );
      const natinf = Natinf(
        numeroNatinf: '12345',
        qualificationInfraction: 'Test',
        dacgSource: 0,
        retraitPoints: 0,
      );

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider(create: (_) => ThemeProvider()),
            ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
            Provider<NatinfRepository>.value(value: repository),
          ],
          child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
        ),
      );

      await tester.pumpAndSettle();

      expect(find.text('Retrait points'), findsNothing);
      await tester.tap(find.text("Afficher plus d'informations…"));
      await tester.pumpAndSettle();
      expect(find.text('Retrait points'), findsOneWidget);
      expect(find.text('0'), findsOneWidget);
    },
  );

  testWidgets(
    'affiche la tentative dans la section principale quand elle est punissable',
    (tester) async {
      final authProvider = await _buildAuthProvider();
      final repository = _StubNatinfRepository(
        unauthorized: false,
        available: false,
        docs: const [],
      );
      const natinf = Natinf(
        numeroNatinf: '12345',
        qualificationInfraction: 'Test',
        dacgSource: 0,
        tentative: true,
      );

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider(create: (_) => ThemeProvider()),
            ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
            Provider<NatinfRepository>.value(value: repository),
          ],
          child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
        ),
      );

      await tester.pumpAndSettle();

      expect(find.text('Tentative'), findsOneWidget);
      expect(find.text('Oui'), findsOneWidget);
    },
  );

  testWidgets('affiche la tentative non punissable dans Plus d’informations', (
    tester,
  ) async {
    final authProvider = await _buildAuthProvider();
    final repository = _StubNatinfRepository(
      unauthorized: false,
      available: false,
      docs: const [],
    );
    const natinf = Natinf(
      numeroNatinf: '12345',
      qualificationInfraction: 'Test',
      dacgSource: 0,
      tentative: false,
    );

    await tester.pumpWidget(
      MultiProvider(
        providers: [
          ChangeNotifierProvider(create: (_) => ThemeProvider()),
          ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
          Provider<NatinfRepository>.value(value: repository),
        ],
        child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
      ),
    );

    await tester.pumpAndSettle();

    expect(find.text('Tentative'), findsNothing);

    await tester.tap(find.text("Afficher plus d'informations…"));
    await tester.pumpAndSettle();

    expect(find.text('Tentative'), findsOneWidget);
    expect(find.text('Non'), findsOneWidget);
  });

  testWidgets('DocPro titles summarize around ">" separators and expand', (
    tester,
  ) async {
    const longTitle =
        'Prévention > Procédure longue > Étape intermédiaire > Conclusion détaillée';
    const noArrowTitle =
        'Très longue procédure DocPro à rallonge pour vérifier la coupure';
    final session = AuthSession.fromTokens(
      access: _token(expMinutes: 60, userId: 1, username: 'retio'),
      refresh: 'refresh',
      refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
    ).copyWith(hasDocPro: true);
    final authProvider = await _buildAuthProvider(session: session);
    final repository = _StubNatinfRepository(
      unauthorized: false,
      available: true,
      docs: const [
        NatinfDocPro(
          id: 1,
          natinf: '12345',
          title: longTitle,
          url: 'https://example.com/docpro/long',
        ),
        NatinfDocPro(
          id: 2,
          natinf: '12345',
          title: noArrowTitle,
          url: 'https://example.com/docpro/long-no-arrow',
        ),
      ],
    );
    const natinf = Natinf(
      numeroNatinf: '12345',
      qualificationInfraction: 'Test',
      dacgSource: 0,
    );

    await tester.pumpWidget(
      MultiProvider(
        providers: [
          ChangeNotifierProvider(create: (_) => ThemeProvider()),
          ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
          Provider<NatinfRepository>.value(value: repository),
        ],
        child: const MaterialApp(home: NatinfDetail(natinf: natinf)),
      ),
    );

    await tester.pumpAndSettle();

    final summarizedWithArrows = _summarizeDocProTitle(longTitle);
    final summarizedNoArrow = _summarizeDocProTitle(noArrowTitle);

    expect(find.text(summarizedWithArrows), findsOneWidget);
    expect(find.text(summarizedNoArrow), findsOneWidget);
    expect(find.text(longTitle), findsNothing);
    expect(find.text(noArrowTitle), findsNothing);

    await tester.tap(find.text(summarizedWithArrows));
    await tester.pumpAndSettle();

    expect(find.text(longTitle), findsOneWidget);
    expect(find.text(noArrowTitle), findsNothing);
    expect(find.byIcon(Icons.launch), findsWidgets);
  });
}

class _StubNatinfRepository extends NatinfRepository {
  _StubNatinfRepository({
    this.unauthorized = false,
    this.available = false,
    this.docs = const [],
  });

  final bool unauthorized;
  final bool available;
  final List<NatinfDocPro> docs;

  @override
  bool isDocsProUnauthorized(String numero) => unauthorized;

  @override
  bool isDocsProAvailable(String numero) => available;

  @override
  Future<List<NatinfDocPro>> fetchDocPro(String numero) async => docs;
}

Future<AuthProvider> _buildAuthProvider({AuthSession? session}) async {
  final storage = InMemoryAuthStorage();
  if (session != null) {
    await storage.save(session);
  }
  final repository = AuthRepository(
    baseUri: Uri.parse('https://natinfo.app/api'),
    storage: storage,
    httpClient: MockClient((request) async => http.Response('{}', 200)),
  );
  final provider = AuthProvider(repository);
  await provider.bootstrap();
  return provider;
}

String _summarizeDocProTitle(String text, {int maxSide = 20}) {
  final parts = text.split('>');
  final arrowCount = parts.length - 1;
  if (arrowCount >= 2) {
    final prefix = parts.first.trim();
    final suffix = parts.last.trim();
    final condensed = '$prefix > ... > $suffix';
    return condensed.length < text.length ? condensed : text;
  }
  if (text.length <= (maxSide * 2) + 7) {
    return text;
  }
  final start = text.substring(0, maxSide);
  final end = text.substring(text.length - maxSide);
  return '$start > ... > $end';
}

String _token({
  required int expMinutes,
  required int userId,
  required String username,
}) {
  final header = base64UrlEncode(utf8.encode('{"alg":"HS256","typ":"JWT"}'));
  final payload = base64UrlEncode(
    utf8.encode(
      '{"exp":${DateTime.now().toUtc().add(Duration(minutes: expMinutes)).millisecondsSinceEpoch ~/ 1000},"user_id":$userId,"username":"$username"}',
    ),
  );
  return '$header.$payload.signature';
}

class _FakePathProviderPlatform extends PathProviderPlatform {
  _FakePathProviderPlatform()
    : tempDir = Directory.systemTemp.createTempSync('natinfo_test_');

  final Directory tempDir;

  @override
  Future<String?> getApplicationDocumentsPath() async => tempDir.path;
}
