import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

void main() {
  group('NatinfRepository source selection', () {
    test('defaults to API source when network allowed', () {
      final registry = SourceRegistry([
        SourceSpec(
          id: 'api-default',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.api,
          uri: Uri.parse('https://default'),
          scope: 'natinf',
          priority: 1,
          requiredAtBuild: false,
          requiresNetwork: true,
          schemaVersion: 1,
        ),
        SourceSpec(
          id: 'asset-fallback',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.asset,
          uri: Uri.parse('asset://natinf.json'),
          scope: 'natinf',
          priority: 2,
          requiredAtBuild: false,
          requiresNetwork: false,
          schemaVersion: 1,
        ),
      ]);
      final resolver = SourceResolver(registry);
      final spec = NatinfRepository().selectNatinfSource(
        resolver: resolver,
        allowNetwork: true,
      );
      expect(spec.id, 'api-default');
      expect(spec.type, SourceType.api);
    });
    test('uses custom URL when provided', () {
      final registry = SourceRegistry([
        SourceSpec(
          id: 'api-default',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.api,
          uri: Uri.parse('https://default'),
          scope: 'natinf',
          priority: 1,
          requiredAtBuild: false,
          requiresNetwork: true,
          schemaVersion: 1,
        ),
      ]);
      final resolver = SourceResolver(registry);
      final spec = NatinfRepository().selectNatinfSource(
        resolver: resolver,
        allowNetwork: true,
        customUrl: 'https://custom',
      );
      expect(spec.id, 'custom-natinf');
      expect(spec.uri.toString(), 'https://custom');
    });

    test('honors preferred source id', () {
      final registry = SourceRegistry([
        SourceSpec(
          id: 'api-default',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.api,
          uri: Uri.parse('https://default'),
          scope: 'natinf',
          priority: 2,
          requiredAtBuild: false,
          requiresNetwork: true,
          schemaVersion: 1,
        ),
        SourceSpec(
          id: 'asset-fallback',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.asset,
          uri: Uri.parse('asset://natinf.json'),
          scope: 'natinf',
          priority: 3,
          requiredAtBuild: false,
          requiresNetwork: false,
          schemaVersion: 1,
        ),
      ]);
      final resolver = SourceResolver(registry);
      final spec = NatinfRepository().selectNatinfSource(
        resolver: resolver,
        allowNetwork: true,
        preferredSourceId: 'asset-fallback',
      );
      expect(spec.id, 'asset-fallback');
    });

    test('falls back to offline source when network disallowed', () {
      final registry = SourceRegistry([
        SourceSpec(
          id: 'api-default',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.api,
          uri: Uri.parse('https://default'),
          scope: 'natinf',
          priority: 1,
          requiredAtBuild: false,
          requiresNetwork: true,
          schemaVersion: 1,
        ),
        SourceSpec(
          id: 'asset-fallback',
          dataset: 'natinf',
          name: 'Infractions NATINFo',
          type: SourceType.asset,
          uri: Uri.parse('asset://natinf.json'),
          scope: 'natinf',
          priority: 2,
          requiredAtBuild: false,
          requiresNetwork: false,
          schemaVersion: 1,
        ),
      ]);
      final resolver = SourceResolver(registry);
      final spec = NatinfRepository().selectNatinfSource(
        resolver: resolver,
        allowNetwork: false,
      );
      expect(spec.id, 'asset-fallback');
    });

    test('uses dedicated categories source when resolving categories', () {
      final registry = SourceRegistry([
        SourceSpec(
          id: 'categories-api',
          dataset: 'natinf-categories',
          name: 'Catégories NATINFo',
          type: SourceType.api,
          uri: Uri.parse('https://categories'),
          scope: 'categories',
          priority: 2,
          requiredAtBuild: false,
          requiresNetwork: true,
          schemaVersion: 1,
        ),
        SourceSpec(
          id: 'categories-asset',
          dataset: 'natinf-categories',
          name: 'Catégories NATINFo',
          type: SourceType.asset,
          uri: Uri.parse('asset://cats.json'),
          scope: 'categories',
          priority: 1,
          requiredAtBuild: false,
          requiresNetwork: false,
          schemaVersion: 1,
        ),
      ]);
      final resolver = SourceResolver(registry);
      final spec = NatinfRepository().selectCategoriesSource(
        resolver: resolver,
        allowNetwork: false,
      );
      expect(spec.id, 'categories-asset');
    });
  });
}
