import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:shared_preferences/shared_preferences.dart';

/// Persists user choices related to source selection.
///
/// Network-allowed flags are clamped by the global build policy (`ALLOW_DOWNLOADS`).
class SourcePreferences {
  SourcePreferences(this._prefs, {bool? globalNetworkAllowed})
    : _globalNetworkAllowed = globalNetworkAllowed ?? AppConfig.allowDownloads;

  final SharedPreferences _prefs;
  final bool _globalNetworkAllowed;

  /// Returns whether network-backed sources are allowed for [dataset].
  bool isNetworkAllowed(String dataset) {
    if (!_globalNetworkAllowed) return false;
    return _prefs.getBool(_networkKey(dataset)) ?? true;
  }

  /// Persists the network allowance for [dataset], clamped by the build policy.
  Future<void> setNetworkAllowed(String dataset, bool allow) async {
    final value = _globalNetworkAllowed ? allow : false;
    await _prefs.setBool(_networkKey(dataset), value);
  }

  /// Whether the build policy forbids allowing network sources altogether.
  bool get isNetworkPolicyLocked => !_globalNetworkAllowed;

  String _networkKey(String dataset) => 'sources.$dataset.allowNetwork';
}
