import 'dart:convert';

import 'package:crypto/crypto.dart';
import 'package:meta/meta.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

/// Result of loading a source.
class SourceLoadResult {
  const SourceLoadResult({
    required this.spec,
    required this.bytes,
    required this.integrity,
  });

  /// Source that produced the data.
  final SourceSpec spec;

  /// Raw payload returned by the source.
  final List<int> bytes;

  /// Integrity status based on the optional checksum declared in the spec.
  final SourceIntegrity integrity;
}

/// Integrity verification result.
class SourceIntegrity {
  const SourceIntegrity({
    required this.status,
    this.expected,
    this.actual,
    this.algorithm,
  });

  /// Status of the checksum validation.
  final ChecksumStatus status;

  /// Expected checksum string from the spec.
  final String? expected;

  /// Computed checksum string for the loaded payload.
  final String? actual;

  /// Algorithm used to compute [actual].
  final String? algorithm;

  /// Whether the data should be considered trustworthy based on checksum.
  bool get isValid => status != ChecksumStatus.mismatch;
}

/// Possible checksum validation results.
enum ChecksumStatus {
  /// No checksum provided in the spec.
  notProvided,

  /// Algorithm not supported, validation skipped.
  skippedUnsupportedAlgorithm,

  /// The computed checksum differs from the expected one.
  mismatch,

  /// The computed checksum matches the expected one.
  match,
}

/// Generic contract for a source adapter.
abstract class SourceAdapter {
  /// Type supported by this adapter.
  SourceType get type;

  /// Loads data for the given [spec].
  Future<SourceLoadResult> load(SourceSpec spec);

  /// Computes and validates the checksum declared in [spec] against [bytes].
  ///
  /// If no checksum is provided, validation is skipped gracefully.
  @protected
  SourceIntegrity verifyIntegrity(SourceSpec spec, List<int> bytes) {
    final expected = spec.checksum;
    final algorithm = spec.checksumAlgo;

    if (expected == null || algorithm == null) {
      return const SourceIntegrity(status: ChecksumStatus.notProvided);
    }

    final digest = _digestFor(algorithm, bytes);
    if (digest == null) {
      return SourceIntegrity(
        status: ChecksumStatus.skippedUnsupportedAlgorithm,
        expected: expected,
        algorithm: algorithm,
      );
    }

    final actual = digest.toString();
    final match = expected.toLowerCase() == actual.toLowerCase();
    return SourceIntegrity(
      status: match ? ChecksumStatus.match : ChecksumStatus.mismatch,
      expected: expected,
      actual: actual,
      algorithm: algorithm,
    );
  }

  Digest? _digestFor(String algorithm, List<int> bytes) {
    switch (algorithm.toLowerCase()) {
      case 'sha256':
        return sha256.convert(bytes);
      case 'sha1':
        return sha1.convert(bytes);
      case 'md5':
        return md5.convert(bytes);
      default:
        return null;
    }
  }
}
