import 'dart:io';

import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

/// Reads dataset payloads from a file on disk (typically a cached download).
class FileSourceAdapter extends SourceAdapter {
  @override
  SourceType get type => SourceType.file;

  @override
  Future<SourceLoadResult> load(SourceSpec spec) async {
    if (spec.type != type) {
      throw ArgumentError.value(
        spec.type,
        'spec.type',
        'FileSourceAdapter only supports $type',
      );
    }

    final file = File(_resolvePath(spec.uri));
    if (!await file.exists()) {
      throw FileSystemException(
        'Source file not found for "${spec.id}"',
        file.path,
      );
    }

    final bytes = await file.readAsBytes();
    final integrity = verifyIntegrity(spec, bytes);
    return SourceLoadResult(spec: spec, bytes: bytes, integrity: integrity);
  }

  String _resolvePath(Uri uri) {
    if (uri.isScheme('file')) {
      return uri.toFilePath();
    }
    if (uri.scheme.isEmpty) {
      return uri.toString();
    }
    throw ArgumentError.value(
      uri.toString(),
      'uri',
      'Unsupported scheme for file source',
    );
  }
}
