import 'package:flutter/services.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

/// Loads dataset payloads from bundled Flutter assets.
class AssetSourceAdapter extends SourceAdapter {
  AssetSourceAdapter({AssetBundle? bundle}) : _bundle = bundle ?? rootBundle;

  final AssetBundle _bundle;

  @override
  SourceType get type => SourceType.asset;

  @override
  Future<SourceLoadResult> load(SourceSpec spec) async {
    if (spec.type != type) {
      throw ArgumentError.value(
        spec.type,
        'spec.type',
        'AssetSourceAdapter only supports $type',
      );
    }

    final key = _resolveKey(spec.uri);
    final data = await _bundle.load(key);
    final bytes = data.buffer.asUint8List();
    final integrity = verifyIntegrity(spec, bytes);
    return SourceLoadResult(spec: spec, bytes: bytes, integrity: integrity);
  }

  String _resolveKey(Uri uri) {
    if (uri.scheme.isEmpty || uri.scheme == 'asset') {
      if (uri.path.isEmpty) {
        if (uri.authority.isNotEmpty) return uri.authority;
        return uri.toString();
      }
      return uri.path.replaceFirst('/', '');
    }
    throw ArgumentError.value(
      uri.toString(),
      'uri',
      'Unsupported scheme for asset source',
    );
  }
}
