import 'dart:async';

import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:natinfo_flutter/app/config/repository_bootstrap.dart';
import 'package:natinfo_flutter/app/state/operational_mode_provider.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/app/navigation/deep_link_service.dart';
import 'package:natinfo_flutter/app/navigation/app_route_observer.dart';
import 'package:provider/provider.dart';

import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/data_initialization_page.dart';

void main() async {
  WidgetsFlutterBinding.ensureInitialized();
  final bootstrap = await RepositoryBootstrap.initialize();
  runApp(
    MyRoot(
      repository: bootstrap.repository,
      themeProvider: bootstrap.themeProvider,
      authProvider: bootstrap.authProvider,
      authRepository: bootstrap.authRepository,
    ),
  );
}

class NoAnimationPageTransitionsBuilder extends PageTransitionsBuilder {
  /// Disables route transition animations.
  @override
  Widget buildTransitions<T>(
    PageRoute<T> route,
    BuildContext context,
    Animation<double> animation,
    Animation<double> secondaryAnimation,
    Widget child,
  ) {
    return child;
  }
}

/// Root widget wiring dependency providers.
class MyRoot extends StatelessWidget {
  final NatinfRepository repository;
  final ThemeProvider themeProvider;
  final AuthProvider authProvider;
  final AuthRepository authRepository;
  const MyRoot({
    super.key,
    required this.repository,
    required this.themeProvider,
    required this.authProvider,
    required this.authRepository,
  });

  @override
  Widget build(BuildContext context) {
    return MultiProvider(
      providers: [
        Provider<NatinfRepository>.value(value: repository),
        Provider<AuthRepository>.value(value: authRepository),
        ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
        ChangeNotifierProvider.value(value: themeProvider),
        ChangeNotifierProvider(create: (_) => OperationalModeProvider()),
        Provider<DeepLinkService>(
          create: (_) {
            final service = DeepLinkService(repository: repository);
            unawaited(service.initialize());
            return service;
          },
          dispose: (_, service) => service.dispose(),
        ),
      ],
      child: MyApp(repository: repository),
    );
  }
}

/// Main application widget.
class MyApp extends StatelessWidget {
  final NatinfRepository repository;
  const MyApp({super.key, required this.repository});

  @override
  Widget build(BuildContext context) {
    final deepLinkService = context.read<DeepLinkService>();
    return Consumer2<ThemeProvider, OperationalModeProvider>(
      builder: (context, themeProvider, opProvider, child) {
        if (opProvider.opMode) {
          SystemChrome.setPreferredOrientations([DeviceOrientation.portraitUp]);
        } else {
          SystemChrome.setPreferredOrientations(DeviceOrientation.values);
        }
        final pageTransitionsTheme =
            opProvider.opMode
                ? PageTransitionsTheme(
                  builders: {
                    TargetPlatform.android: NoAnimationPageTransitionsBuilder(),
                    TargetPlatform.iOS: NoAnimationPageTransitionsBuilder(),
                  },
                )
                : ThemeData.light().pageTransitionsTheme;
        return MaterialApp(
          navigatorObservers: [appRouteObserver],
          title: 'NATINFo+',
          theme: ThemeData.light().copyWith(
            primaryColor: themeProvider.primaryColor,
            appBarTheme: AppBarTheme(
              color: themeProvider.primaryColor,
              foregroundColor: themeProvider.primaryTextColor,
            ),
            pageTransitionsTheme: pageTransitionsTheme,
          ),
          darkTheme: ThemeData.dark().copyWith(
            primaryColor: themeProvider.primaryColor,
            appBarTheme: AppBarTheme(
              color: themeProvider.primaryColor,
              foregroundColor: themeProvider.primaryTextColor,
            ),
            pageTransitionsTheme: pageTransitionsTheme,
          ),
          themeMode:
              themeProvider.isDarkMode ? ThemeMode.dark : ThemeMode.light,
          navigatorKey: deepLinkService.navigatorKey,
          home: DataInitializationScreen(repository: repository),
        );
      },
    );
  }
}
