import 'dart:math';
import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';

/// ========= ENUMS & TYPES =========

enum SpeedCalcMode {
  cinemometre, // radar
  moyenne, // vitesse moyenne
  isochrone, // temps théorique
  distanceArret, // distance/vitesse d'arrêt
  circonstances, // "eu égard aux circonstances"
  incitation, // incitation employeur/donneur d'ordre
}

enum Domain {
  route, // véhicule à moteur
  sansMoteur, // cycles etc.
  pietonsTrottoirs, // aires piétonnes / trottoirs / contre-allées
  eauxInterieures, // général + remous
  rhenan, // navigation rhénane
  moselle, // Moselle internationale
}

enum CineType { fixe, mobile }

enum DriverType { normal, eleve }

enum VmaBand { le50, gt50 }

enum EauxInfraction { excesVitesse, remousGeneral }

enum RemousContext {
  presBateauAmarre,
  droitBateauProtege,
  presBac,
  devantEntreePort,
  presBateauxStationnement,
  secteurReglemente,
  bateauPassagers,
}

enum RhenanType { general, passagers, matDangereuses, proximiteMatDang }

enum PietonLieu { airePietonne, trottoirContreAllee }

enum VehiculePieton { cycle, cyclomoteur }

enum AccidentType { aucun, blessuresLe3Mois, blessuresGt3Mois, homicide }

enum SurfaceCondition { sec, mouille }

enum DistanceCalcMode { fromSpeed, fromDistance }

enum IncitationActeur { employeur, donneurOrdre }

enum IncitationTransport { personnes, marchandises, matDangOuExceptionnel }

/// ========= PAGE =========

class CalculateurVitessePage extends StatefulWidget {
  const CalculateurVitessePage({super.key});
  @override
  State<CalculateurVitessePage> createState() => _CalculateurVitessePageState();
}

class _CalculateurVitessePageState extends State<CalculateurVitessePage> {
  // Sélections
  SpeedCalcMode _mode = SpeedCalcMode.cinemometre;
  Domain _domain = Domain.route;

  // Cinémomètre / génériques vitesse
  final _vitesseCtrl = TextEditingController();
  final _vmaCtrl = TextEditingController();
  CineType _cineType = CineType.fixe;
  DriverType _driver = DriverType.normal;
  bool _redevable = false;
  bool _recidive50 = false;
  AccidentType _accident = AccidentType.aucun;

  // Moyenne
  final _pk1Ctrl = TextEditingController();
  final _pk2Ctrl = TextEditingController();
  TimeOfDay? _tDep;
  TimeOfDay? _tArr;

  // Sans moteur
  final _vitesseSansMoteurCtrl = TextEditingController();
  final _vmaSansMoteurCtrl = TextEditingController();

  // Piétons/trottoirs
  PietonLieu _pietonLieu = PietonLieu.airePietonne;
  VehiculePieton _vehPieton = VehiculePieton.cycle;

  // Eaux / rhénan / moselle
  EauxInfraction _eauxInfraction = EauxInfraction.excesVitesse;
  RemousContext _remousContext = RemousContext.presBateauAmarre;
  RhenanType _rhenanType = RhenanType.general;

  // Distance d'arrêt
  DistanceCalcMode _distMode = DistanceCalcMode.fromSpeed;
  SurfaceCondition _surface = SurfaceCondition.sec;
  final _distanceCtrl = TextEditingController();

  // Incitation
  IncitationActeur _incitActeur = IncitationActeur.employeur;
  IncitationTransport _incitTransport = IncitationTransport.marchandises;

  // Résultats
  String? _resultat, _explication, _classification, _error;
  List<String>? _natinfCodes;
  List<Natinf>? _natinfResults;
  bool _loading = false;

  static const _urlCine =
      'https://www.legifrance.gouv.fr/loda/id/LEGIARTI000020779878/2009-06-24';
  static const _urlDistanceArret =
      'https://media.education.gouv.fr/file/Securite/92/3/07assr_3maths_distancearret_108923.pdf';

  @override
  void initState() {
    super.initState();
    MatomoService().trackPage(
      title: 'Calculateur de vitesse',
      path: '/tools/vitesse/',
    );
    for (final c in [
      _vitesseCtrl,
      _vmaCtrl,
      _pk1Ctrl,
      _pk2Ctrl,
      _distanceCtrl,
      _vitesseSansMoteurCtrl,
      _vmaSansMoteurCtrl,
    ]) {
      c.addListener(() => setState(() {}));
    }
  }

  @override
  void dispose() {
    _vitesseCtrl.dispose();
    _vmaCtrl.dispose();
    _pk1Ctrl.dispose();
    _pk2Ctrl.dispose();
    _distanceCtrl.dispose();
    _vitesseSansMoteurCtrl.dispose();
    _vmaSansMoteurCtrl.dispose();
    super.dispose();
  }

  double? _parse(String s) => double.tryParse(s.replaceAll(',', '.'));

  /// ========= COMPATIBILITÉ MODE / DOMAINE =========

  static const Map<SpeedCalcMode, Set<Domain>> _allowedDomainsByMode = {
    SpeedCalcMode.cinemometre: {
      Domain.route,
      Domain.sansMoteur,
      Domain.pietonsTrottoirs,
      Domain.eauxInterieures,
      Domain.rhenan,
      Domain.moselle,
    },
    SpeedCalcMode.moyenne: {
      Domain.route,
      Domain.sansMoteur,
      Domain.eauxInterieures,
      Domain.moselle,
    },
    SpeedCalcMode.isochrone: {Domain.route},
    SpeedCalcMode.distanceArret: {Domain.route},
    SpeedCalcMode.circonstances: {Domain.route},
    SpeedCalcMode.incitation: {Domain.route},
  };

  static const Map<Domain, Set<SpeedCalcMode>> _allowedModesByDomain = {
    Domain.route: {
      SpeedCalcMode.cinemometre,
      SpeedCalcMode.moyenne,
      SpeedCalcMode.isochrone,
      SpeedCalcMode.distanceArret,
      SpeedCalcMode.circonstances,
      SpeedCalcMode.incitation,
    },
    Domain.sansMoteur: {SpeedCalcMode.cinemometre, SpeedCalcMode.moyenne},
    Domain.pietonsTrottoirs: {SpeedCalcMode.cinemometre},
    Domain.eauxInterieures: {SpeedCalcMode.cinemometre, SpeedCalcMode.moyenne},
    Domain.rhenan: {SpeedCalcMode.cinemometre},
    Domain.moselle: {SpeedCalcMode.cinemometre, SpeedCalcMode.moyenne},
  };

  bool _isDomainAllowedForMode(Domain d, SpeedCalcMode m) =>
      _allowedDomainsByMode[m]!.contains(d);
  bool _isModeAllowedForDomain(SpeedCalcMode m, Domain d) =>
      _allowedModesByDomain[d]!.contains(m);

  void _ensureCompatibilityAfterChange({bool changedMode = false}) {
    if (!_isDomainAllowedForMode(_domain, _mode)) {
      _domain = _allowedDomainsByMode[_mode]!.first;
    }
    if (!_isModeAllowedForDomain(_mode, _domain)) {
      _mode = _allowedModesByDomain[_domain]!.first;
    }
    _clearOutputs();
    _autoQualifyMaybe();
  }

  /// ========= LOGIQUE D’AFFICHAGE =========

  bool _needsSpeedFields() {
    if (_mode == SpeedCalcMode.moyenne ||
        _mode == SpeedCalcMode.isochrone ||
        _mode == SpeedCalcMode.distanceArret ||
        _mode == SpeedCalcMode.circonstances ||
        _mode == SpeedCalcMode.incitation) {
      return false;
    }
    switch (_domain) {
      case Domain.route:
      case Domain.sansMoteur:
        return true;
      case Domain.pietonsTrottoirs:
        return false;
      case Domain.eauxInterieures:
        return _eauxInfraction == EauxInfraction.excesVitesse;
      case Domain.rhenan:
        return false;
      case Domain.moselle:
        return true;
    }
  }

  bool _showActionButton() {
    if (_mode == SpeedCalcMode.circonstances ||
        _mode == SpeedCalcMode.incitation) {
      return false;
    }
    if (_mode == SpeedCalcMode.cinemometre && !_needsSpeedFields()) {
      return false;
    }
    return true;
  }

  void _autoQualifyMaybe() {
    if (!_showActionButton()) {
      Future.microtask(_calculate);
    }
  }

  /// ========= VALIDATION (désactivation bouton) =========

  bool _canCalculate() {
    switch (_mode) {
      case SpeedCalcMode.cinemometre:
        if (!_needsSpeedFields()) return false; // auto
        return _parse(_vitesseCtrl.text) != null &&
            _parse(_vmaCtrl.text) != null;

      case SpeedCalcMode.moyenne:
        // PK + heures obligatoires ; VMA FACULTATIVE
        final pkOk =
            _parse(_pk1Ctrl.text) != null && _parse(_pk2Ctrl.text) != null;
        final hOk = _tDep != null && _tArr != null;
        return pkOk && hOk;

      case SpeedCalcMode.isochrone:
        return _parse(_pk1Ctrl.text) != null &&
            _parse(_pk2Ctrl.text) != null &&
            _parse(_vitesseCtrl.text) != null;

      case SpeedCalcMode.distanceArret:
        return _distMode == DistanceCalcMode.fromSpeed
            ? _parse(_vitesseCtrl.text) != null
            : _parse(_distanceCtrl.text) != null;

      case SpeedCalcMode.circonstances:
      case SpeedCalcMode.incitation:
        return false; // pas de bouton
    }
  }

  /// ========= CALCULS & MAPPINGS =========

  (int retenue, String details) _retenueCinemometre(double v) {
    double raw;
    String exp;
    if (_cineType == CineType.fixe) {
      if (v < 100) {
        raw = v - 5;
        exp = '${v.toStringAsFixed(0)} − 5 = ${raw.toStringAsFixed(2)}';
      } else {
        raw = v * 0.95;
        exp = '${v.toStringAsFixed(0)} − 5 % = ${raw.toStringAsFixed(2)}';
      }
    } else {
      if (v < 100) {
        raw = v - 10;
        exp = '${v.toStringAsFixed(0)} − 10 = ${raw.toStringAsFixed(2)}';
      } else {
        raw = v * 0.90;
        exp = '${v.toStringAsFixed(0)} − 10 % = ${raw.toStringAsFixed(2)}';
      }
    }
    final retenue = raw.floor();
    return (retenue, '$exp → vitesse retenue = $retenue km/h');
  }

  double _vitesseMoyKmH(double pk1, double pk2, TimeOfDay t1, TimeOfDay t2) {
    final dKm = (pk2 - pk1).abs();
    final s1 = t1.hour * 3600 + t1.minute * 60;
    final s2 = t2.hour * 3600 + t2.minute * 60;
    final dS = (s2 - s1).toDouble();
    if (dS <= 0) throw Exception('L’arrivée doit être après le départ.');
    return dKm / (dS / 3600.0);
  }

  (double d, String details) _distanceArretDepuisVitesse(double vKmH) {
    final vms = vKmH / 3.6;
    final dReaction = vms * 1.0;
    final k = _surface == SurfaceCondition.sec ? 0.08 : 0.14;
    final dFreinage = k * vms * vms;
    final total = dReaction + dFreinage;
    final details =
        'v = ${vKmH.toStringAsFixed(0)} km/h → vₘₛ = ${vms.toStringAsFixed(2)} m/s\n'
        'Réaction : 1 s × vₘₛ = ${dReaction.toStringAsFixed(2)} m\n'
        'Freinage : k × vₘₛ² = ${k.toStringAsFixed(2)} × ${vms.toStringAsFixed(2)}² = ${dFreinage.toStringAsFixed(2)} m';
    return (total, details);
  }

  (double vKmH, String details) _vitesseDepuisDistanceArret(double d) {
    final k = _surface == SurfaceCondition.sec ? 0.08 : 0.14;
    final a = k, b = 1.0, c = -d;
    final disc = b * b - 4 * a * c;
    final vms = (-b + sqrt(disc)) / (2 * a);
    final v = vms * 3.6;
    final details =
        'Équation : k·vₘₛ² + vₘₛ - d = 0\n'
        'Solution vₘₛ = ${vms.toStringAsFixed(2)} m/s → v = ${v.toStringAsFixed(2)} km/h';
    return (v, details);
  }

  List<String> _natinfRoute({
    required int retenue,
    required int vma,
    required DriverType driver,
    required bool redevable,
    required bool recidive50,
    required AccidentType accident,
  }) {
    final delta = retenue - vma;
    if (delta <= 0) return <String>[];
    final vmaBand = vma <= 50 ? VmaBand.le50 : VmaBand.gt50;

    final accidentCodes = <String>[];
    if (delta >= 50) {
      switch (accident) {
        case AccidentType.blessuresLe3Mois:
          accidentCodes.add('24001');
          break;
        case AccidentType.blessuresGt3Mois:
          accidentCodes.add('24002');
          break;
        case AccidentType.homicide:
          accidentCodes.add('24003');
          break;
        case AccidentType.aucun:
          break;
      }
    }

    if (redevable) {
      if (delta >= 50) return ['22052', ...accidentCodes];
      if (delta >= 20) return ['25388'];
      return [vmaBand == VmaBand.gt50 ? '25390' : '25389'];
    }

    if (driver == DriverType.eleve) {
      if (delta >= 50)
        return [recidive50 ? '22037' : '21540', ...accidentCodes];
      if (delta >= 40) return ['21541'];
      if (delta >= 30) return ['12927'];
      if (delta >= 20) return ['25391'];
      if (delta >= 5) return [vmaBand == VmaBand.gt50 ? '25393' : '25392'];
      return [vmaBand == VmaBand.gt50 ? '35274' : '35273'];
    } else {
      if (delta >= 50)
        return [recidive50 ? '22037' : '21526', ...accidentCodes];
      if (delta >= 40) return ['21527'];
      if (delta >= 30) return ['11301'];
      if (delta >= 20) return ['11302'];
      if (delta >= 5) return [vmaBand == VmaBand.gt50 ? '25387' : '25386'];
      return [vmaBand == VmaBand.gt50 ? '35274' : '35273'];
    }
  }

  List<String> _natinfSansMoteur({required int retenue, required int vma}) {
    return retenue > vma ? ['6189'] : <String>[];
  }

  List<String> _natinfPietonTrottoir() {
    if (_pietonLieu == PietonLieu.airePietonne) return ['22923'];
    return [_vehPieton == VehiculePieton.cycle ? '22924' : '22925'];
  }

  List<String> _natinfEaux({required int? retenue, required int? vma}) {
    if (_domain == Domain.eauxInterieures) {
      if (_eauxInfraction == EauxInfraction.excesVitesse) {
        if (retenue != null && vma != null && retenue > vma) return ['23694'];
        return <String>[];
      } else {
        switch (_remousContext) {
          case RemousContext.presBateauAmarre:
            return ['30516'];
          case RemousContext.droitBateauProtege:
            return ['30522'];
          case RemousContext.presBac:
            return ['30519'];
          case RemousContext.devantEntreePort:
            return ['30517'];
          case RemousContext.presBateauxStationnement:
            return ['30518'];
          case RemousContext.secteurReglemente:
            return ['30520'];
          case RemousContext.bateauPassagers:
            return ['30455'];
        }
      }
    } else if (_domain == Domain.rhenan) {
      switch (_rhenanType) {
        case RhenanType.general:
          return ['33188'];
        case RhenanType.passagers:
          return ['33189'];
        case RhenanType.matDangereuses:
          return ['33190'];
        case RhenanType.proximiteMatDang:
          return ['33191'];
      }
    } else if (_domain == Domain.moselle) {
      if (retenue != null && vma != null && retenue > vma) return ['33699'];
      return <String>[];
    }
    return <String>[];
  }

  List<String> _natinfCirconstances({required bool redevable}) => [
    redevable ? '32994' : '213',
  ];

  List<String> _natinfIncitation() {
    if (_incitActeur == IncitationActeur.employeur) {
      switch (_incitTransport) {
        case IncitationTransport.marchandises:
          return ['11830'];
        case IncitationTransport.personnes:
          return ['11834'];
        case IncitationTransport.matDangOuExceptionnel:
          return ['11832'];
      }
    } else {
      switch (_incitTransport) {
        case IncitationTransport.marchandises:
          return ['11831'];
        case IncitationTransport.personnes:
          return <String>[];
        case IncitationTransport.matDangOuExceptionnel:
          return ['11833'];
      }
    }
  }

  /// ========= ACTION PRINCIPALE =========

  Future<void> _calculate() async {
    setState(() {
      _error = _resultat = _explication = _classification = null;
      _natinfCodes = _natinfResults = null;
    });

    if (!_isDomainAllowedForMode(_domain, _mode) ||
        !_isModeAllowedForDomain(_mode, _domain)) {
      setState(() => _error = 'Combinaison mode/domaine non applicable.');
      return;
    }

    try {
      List<String> codes = <String>[];
      String? exp, classif;
      int? retenue, vma;

      switch (_mode) {
        case SpeedCalcMode.cinemometre:
          if (!_needsSpeedFields()) {
            if (_domain == Domain.pietonsTrottoirs) {
              codes = _natinfPietonTrottoir();
            } else {
              codes = _natinfEaux(retenue: null, vma: null);
            }
            classif = codes.isEmpty ? 'Aucune infraction' : 'Contraventionnel';
            _resultat = 'Qualification sélectionnée';
            exp = 'Aucune saisie de vitesse requise pour ce contexte.';
            break;
          }
          final v = _parse(_vitesseCtrl.text);
          final lim = _parse(_vmaCtrl.text);
          if (v == null || lim == null) return; // bouton normalement désactivé
          final out = _retenueCinemometre(v);
          retenue = out.$1;
          vma = lim.round();
          exp = out.$2;
          _resultat = 'Vitesse retenue : $retenue km/h (VMA $vma)';

          if (_domain == Domain.route) {
            codes = _natinfRoute(
              retenue: retenue,
              vma: vma,
              driver: _driver,
              redevable: _redevable,
              recidive50: _recidive50,
              accident: _accident,
            );
            classif =
                codes.isEmpty
                    ? 'Aucune infraction'
                    : ((retenue - vma) >= 50
                        ? 'Contraventionnel / Délictuel (selon cas)'
                        : 'Contraventionnel');
          } else if (_domain == Domain.sansMoteur) {
            codes = _natinfSansMoteur(retenue: retenue, vma: vma);
            classif = codes.isEmpty ? 'Aucune infraction' : 'Contraventionnel';
          } else if (_domain == Domain.moselle ||
              _domain == Domain.eauxInterieures) {
            codes = _natinfEaux(retenue: retenue, vma: vma);
            classif = codes.isEmpty ? 'Aucune infraction' : 'Contraventionnel';
          }
          break;

        case SpeedCalcMode.moyenne:
          final pk1 = _parse(_pk1Ctrl.text);
          final pk2 = _parse(_pk2Ctrl.text);
          if (pk1 == null || pk2 == null || _tDep == null || _tArr == null)
            return;
          final v = _vitesseMoyKmH(pk1, pk2, _tDep!, _tArr!);
          retenue = v.floor();
          final dKm = (pk2 - pk1).abs();
          exp =
              'Distance = ${dKm.toStringAsFixed(2)} km, '
              'Durée = ${_fmtTimeDelta(_tDep!, _tArr!)} → '
              'Vitesse moyenne = ${v.toStringAsFixed(2)} km/h (retenue = $retenue)';

          // VMA facultative (qualif seulement si disponible)
          double? lim;
          if (_domain == Domain.sansMoteur) {
            lim = _parse(_vmaSansMoteurCtrl.text);
          } else {
            lim = _parse(_vmaCtrl.text);
          }

          if (lim == null) {
            _resultat = 'Vitesse moyenne retenue : $retenue km/h';
            classif = null;
            codes = <String>[];
          } else {
            vma = lim.round();
            _resultat = 'Vitesse moyenne retenue : $retenue km/h (VMA $vma)';
            if (_domain == Domain.route) {
              codes = _natinfRoute(
                retenue: retenue,
                vma: vma!,
                driver: _driver,
                redevable: _redevable,
                recidive50: _recidive50,
                accident: _accident,
              );
              classif =
                  codes.isEmpty
                      ? 'Aucune infraction'
                      : ((retenue - vma!) >= 50
                          ? 'Contraventionnel / Délictuel (selon cas)'
                          : 'Contraventionnel');
            } else if (_domain == Domain.sansMoteur) {
              codes = _natinfSansMoteur(retenue: retenue, vma: vma!);
              classif =
                  codes.isEmpty ? 'Aucune infraction' : 'Contraventionnel';
            } else if (_domain == Domain.moselle ||
                _domain == Domain.eauxInterieures) {
              codes = _natinfEaux(retenue: retenue, vma: vma);
              classif =
                  codes.isEmpty ? 'Aucune infraction' : 'Contraventionnel';
            }
          }
          break;

        case SpeedCalcMode.isochrone:
          final pk1i = _parse(_pk1Ctrl.text);
          final pk2i = _parse(_pk2Ctrl.text);
          final vi = _parse(_vitesseCtrl.text);
          if (pk1i == null || pk2i == null || vi == null) return;
          final dKm = (pk2i - pk1i).abs();
          final secs = (dKm / vi * 3600).round();
          final h = secs ~/ 3600, m = (secs % 3600) ~/ 60, s = secs % 60;
          _resultat = 'Temps théorique : ${_two(h)}:${_two(m)}:${_two(s)}';
          exp =
              'Distance = ${dKm.toStringAsFixed(2)} km, '
              'Vitesse = ${vi.toStringAsFixed(2)} km/h → Temps = $secs s';
          classif = null;
          codes = <String>[];
          break;

        case SpeedCalcMode.distanceArret:
          if (_distMode == DistanceCalcMode.fromSpeed) {
            final vds = _parse(_vitesseCtrl.text);
            if (vds == null) return;
            final out = _distanceArretDepuisVitesse(vds);
            _resultat = 'Distance d’arrêt : ${out.$1.toStringAsFixed(2)} m';
            exp = out.$2;
          } else {
            final d = _parse(_distanceCtrl.text);
            if (d == null) return;
            final out = _vitesseDepuisDistanceArret(d);
            _resultat = 'Vitesse estimée : ${out.$1.toStringAsFixed(2)} km/h';
            exp = out.$2;
          }
          classif = null;
          codes = <String>[];
          break;

        case SpeedCalcMode.circonstances:
          codes = _natinfCirconstances(redevable: _redevable);
          _resultat = 'Vitesse excessive eu égard aux circonstances';
          exp =
              _redevable ? 'Redevable de l’amende (32994)' : 'Conducteur (213)';
          classif = 'Contraventionnel';
          break;

        case SpeedCalcMode.incitation:
          codes = _natinfIncitation();
          _resultat = "Incitation à l’excès de vitesse";
          exp =
              _incitActeur == IncitationActeur.employeur
                  ? 'Acteur : Employeur'
                  : "Acteur : Donneur d’ordre";
          classif =
              codes.isNotEmpty
                  ? 'Contraventionnel'
                  : 'Aucune infraction (combinaison non prévue)';
          break;
      }

      setState(() {
        _explication = exp;
        _classification = classif;
        _natinfCodes = codes.toSet().toList();
      });

      if (codes.isNotEmpty) {
        await _fetchNatinfDetails(codes);
      }
    } catch (e) {
      setState(() => _error = 'Erreur : ${e.toString()}');
    }
  }

  Future<void> _fetchNatinfDetails(List<String> codes) async {
    setState(() {
      _loading = true;
      _natinfResults = null;
    });
    final results = <Natinf>[];
    for (final code in codes) {
      final n = await DatabaseHelper().getNatinfByNumero(code);
      if (n != null) results.add(n);
    }
    if (!mounted) return;
    setState(() {
      _loading = false;
      _natinfResults = results;
    });
  }

  /// ========= HELPERS UI =========

  void _clearOutputs() {
    _resultat = _explication = _classification = _error = null;
    _natinfCodes = _natinfResults = null;
  }

  String _two(int n) => n.toString().padLeft(2, '0');

  String _fmtTimeDelta(TimeOfDay a, TimeOfDay b) {
    final s1 = a.hour * 3600 + a.minute * 60;
    final s2 = b.hour * 3600 + b.minute * 60;
    final d = s2 - s1;
    final h = d ~/ 3600, m = (d % 3600) ~/ 60, s = d % 60;
    return '${_two(h)}:${_two(m)}:${_two(s)}';
  }

  Future<void> _pickTime(bool depart) async {
    final t = await showTimePicker(
      context: context,
      initialTime: TimeOfDay.now(),
    );
    if (t != null) setState(() => depart ? _tDep = t : _tArr = t);
  }

  InputDecoration _dec(String label, {String? hint}) => InputDecoration(
    labelText: label,
    hintText: hint,
    border: const OutlineInputBorder(),
    contentPadding: const EdgeInsets.symmetric(horizontal: 12, vertical: 10),
  );

  Widget _disclaimerCard() {
    final cs = Theme.of(context).colorScheme;
    return Container(
      decoration: BoxDecoration(
        color: cs.surfaceVariant,
        borderRadius: BorderRadius.circular(10),
      ),
      padding: const EdgeInsets.all(12),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: const [
          Icon(Icons.info_outline),
          SizedBox(width: 8),
          Expanded(
            child: Text(
              "Ce calculateur fournit une orientation à partir d’un sous-ensemble de NATINF et de règles usuelles. "
              "Selon les circonstances (multi-infractions, contextes spécifiques), d’autres qualifications peuvent s’appliquer. "
              "Vérifiez toujours la qualification retenue et les textes en vigueur.",
              style: TextStyle(height: 1.25),
            ),
          ),
        ],
      ),
    );
  }

  List<DropdownMenuItem<Domain>> _domainItemsFiltered() {
    final allowed = _allowedDomainsByMode[_mode]!;
    return Domain.values
        .where((d) => allowed.contains(d))
        .map((d) => DropdownMenuItem(value: d, child: Text(_labelDomain(d))))
        .toList();
  }

  List<DropdownMenuItem<SpeedCalcMode>> _modeItemsFiltered() {
    final allowed = _allowedModesByDomain[_domain]!;
    return SpeedCalcMode.values
        .where((m) => allowed.contains(m))
        .map((m) => DropdownMenuItem(value: m, child: Text(_labelMode(m))))
        .toList();
  }

  String _labelMode(SpeedCalcMode m) {
    switch (m) {
      case SpeedCalcMode.cinemometre:
        return 'Cinémomètre (radar)';
      case SpeedCalcMode.moyenne:
        return 'Vitesse moyenne';
      case SpeedCalcMode.isochrone:
        return 'Isochrone (temps théorique)';
      case SpeedCalcMode.distanceArret:
        return 'Distance d’arrêt / vitesse';
      case SpeedCalcMode.circonstances:
        return 'Vitesse excessive (circonstances)';
      case SpeedCalcMode.incitation:
        return "Incitation à l’excès de vitesse";
    }
  }

  String _labelDomain(Domain d) {
    switch (d) {
      case Domain.route:
        return 'Route (véhicule à moteur)';
      case Domain.sansMoteur:
        return 'Sans moteur (cycle, etc.)';
      case Domain.pietonsTrottoirs:
        return 'Piétons / trottoirs / aires';
      case Domain.eauxInterieures:
        return 'Eaux intérieures (général/remous)';
      case Domain.rhenan:
        return 'Navigation rhénane';
      case Domain.moselle:
        return 'Moselle internationale';
    }
  }

  /// ========= BUILD =========

  @override
  Widget build(BuildContext context) {
    if (!_isDomainAllowedForMode(_domain, _mode) ||
        !_isModeAllowedForDomain(_mode, _domain)) {
      WidgetsBinding.instance.addPostFrameCallback((_) {
        if (!mounted) return;
        setState(() => _ensureCompatibilityAfterChange());
      });
    }

    final showAction = _showActionButton();
    final needsSpeed = _needsSpeedFields();
    final canCalc = _canCalculate();

    return Scaffold(
      appBar: AppBar(title: const Text('Calculateur de vitesse')),
      body: LayoutBuilder(
        builder: (context, c) {
          final isWide = c.maxWidth >= 640;
          final topRow =
              isWide
                  ? Row(
                    children: [
                      Expanded(
                        child: DropdownButtonFormField<SpeedCalcMode>(
                          value: _mode,
                          isExpanded: true,
                          decoration: _dec('Mode'),
                          items: _modeItemsFiltered(),
                          onChanged: (v) {
                            setState(() {
                              _mode = v!;
                              _ensureCompatibilityAfterChange(
                                changedMode: true,
                              );
                            });
                          },
                        ),
                      ),
                      const SizedBox(width: 12),
                      Expanded(
                        child: DropdownButtonFormField<Domain>(
                          value: _domain,
                          isExpanded: true,
                          decoration: _dec('Domaine'),
                          items: _domainItemsFiltered(),
                          onChanged: (v) {
                            setState(() {
                              _domain = v!;
                              _ensureCompatibilityAfterChange();
                            });
                          },
                        ),
                      ),
                    ],
                  )
                  : Column(
                    children: [
                      DropdownButtonFormField<SpeedCalcMode>(
                        value: _mode,
                        isExpanded: true,
                        decoration: _dec('Mode'),
                        items: _modeItemsFiltered(),
                        onChanged: (v) {
                          setState(() {
                            _mode = v!;
                            _ensureCompatibilityAfterChange(changedMode: true);
                          });
                        },
                      ),
                      const SizedBox(height: 12),
                      DropdownButtonFormField<Domain>(
                        value: _domain,
                        isExpanded: true,
                        decoration: _dec('Domaine'),
                        items: _domainItemsFiltered(),
                        onChanged: (v) {
                          setState(() {
                            _domain = v!;
                            _ensureCompatibilityAfterChange();
                          });
                        },
                      ),
                    ],
                  );

          return SingleChildScrollView(
            padding: const EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.stretch,
              children: [
                topRow,
                const SizedBox(height: 16),

                // Cinémomètre : réglages spécifique route
                if (_mode == SpeedCalcMode.cinemometre &&
                    _domain == Domain.route) ...[
                  const Text(
                    'Cinémomètre',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  Row(
                    children: [
                      Expanded(
                        child: RadioListTile<CineType>(
                          title: const Text('Fixe'),
                          value: CineType.fixe,
                          groupValue: _cineType,
                          onChanged: (v) => setState(() => _cineType = v!),
                        ),
                      ),
                      Expanded(
                        child: RadioListTile<CineType>(
                          title: const Text('Mobile'),
                          value: CineType.mobile,
                          groupValue: _cineType,
                          onChanged: (v) => setState(() => _cineType = v!),
                        ),
                      ),
                    ],
                  ),
                ],

                // Champs vitesse/VMA génériques si nécessaires
                if (_mode == SpeedCalcMode.cinemometre && needsSpeed) ...[
                  TextField(
                    controller: _vitesseCtrl,
                    decoration: _dec('Vitesse mesurée (km/h)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  TextField(
                    controller: _vmaCtrl,
                    decoration: _dec('Vitesse maximale autorisée (VMA)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                ],

                if (_mode == SpeedCalcMode.cinemometre &&
                    _domain == Domain.route) ...[
                  const SizedBox(height: 8),
                  DropdownButtonFormField<DriverType>(
                    value: _driver,
                    isExpanded: true,
                    decoration: _dec('Type de conducteur'),
                    items: const [
                      DropdownMenuItem(
                        value: DriverType.normal,
                        child: Text('Conducteur normal'),
                      ),
                      DropdownMenuItem(
                        value: DriverType.eleve,
                        child: Text('Élève conducteur'),
                      ),
                    ],
                    onChanged: (v) => setState(() => _driver = v!),
                  ),
                  SwitchListTile(
                    value: _redevable,
                    onChanged: (v) => setState(() => _redevable = v),
                    title: const Text(
                      'Redevable de l’amende (titulaire CG / personne morale)',
                    ),
                  ),
                  SwitchListTile(
                    value: _recidive50,
                    onChanged: (v) => setState(() => _recidive50 = v),
                    title: const Text('Récidive (si dépassement ≥ 50 km/h)'),
                  ),
                  DropdownButtonFormField<AccidentType>(
                    value: _accident,
                    isExpanded: true,
                    decoration: _dec('Accident corporel (optionnel)'),
                    items: const [
                      DropdownMenuItem(
                        value: AccidentType.aucun,
                        child: Text('Aucun'),
                      ),
                      DropdownMenuItem(
                        value: AccidentType.blessuresLe3Mois,
                        child: Text('Blessures ≤ 3 mois'),
                      ),
                      DropdownMenuItem(
                        value: AccidentType.blessuresGt3Mois,
                        child: Text('Blessures > 3 mois'),
                      ),
                      DropdownMenuItem(
                        value: AccidentType.homicide,
                        child: Text('Homicide involontaire'),
                      ),
                    ],
                    onChanged: (v) => setState(() => _accident = v!),
                  ),
                ],

                // Sans moteur
                if ((_mode == SpeedCalcMode.cinemometre ||
                        _mode == SpeedCalcMode.moyenne) &&
                    _domain == Domain.sansMoteur) ...[
                  const Divider(height: 24),
                  const Text(
                    'Sans moteur (cycle, etc.)',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  TextField(
                    controller: _vitesseSansMoteurCtrl,
                    decoration: _dec('Vitesse constatée (km/h)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  TextField(
                    controller: _vmaSansMoteurCtrl,
                    decoration: _dec(
                      'Vitesse maximale autorisée (VMA) – optionnel pour la moyenne',
                    ),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                ],

                // Piétons/trottoirs (qualification directe)
                if (_domain == Domain.pietonsTrottoirs) ...[
                  const Divider(height: 24),
                  const Text(
                    'Piétons / trottoirs / aires',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  DropdownButtonFormField<PietonLieu>(
                    value: _pietonLieu,
                    isExpanded: true,
                    decoration: _dec('Lieu'),
                    items: const [
                      DropdownMenuItem(
                        value: PietonLieu.airePietonne,
                        child: Text('Aire piétonne'),
                      ),
                      DropdownMenuItem(
                        value: PietonLieu.trottoirContreAllee,
                        child: Text('Trottoir / contre-allée (hors agglo)'),
                      ),
                    ],
                    onChanged: (v) {
                      setState(() => _pietonLieu = v!);
                      _autoQualifyMaybe();
                    },
                  ),
                  DropdownButtonFormField<VehiculePieton>(
                    value: _vehPieton,
                    isExpanded: true,
                    decoration: _dec('Véhicule'),
                    items: const [
                      DropdownMenuItem(
                        value: VehiculePieton.cycle,
                        child: Text('Cycle'),
                      ),
                      DropdownMenuItem(
                        value: VehiculePieton.cyclomoteur,
                        child: Text('Cyclomoteur'),
                      ),
                    ],
                    onChanged: (v) {
                      setState(() => _vehPieton = v!);
                      _autoQualifyMaybe();
                    },
                  ),
                ],

                // Eaux intérieures
                if (_domain == Domain.eauxInterieures) ...[
                  const Divider(height: 24),
                  const Text(
                    'Voies navigables – eaux intérieures',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  DropdownButtonFormField<EauxInfraction>(
                    value: _eauxInfraction,
                    isExpanded: true,
                    decoration: _dec('Type d’infraction'),
                    items: const [
                      DropdownMenuItem(
                        value: EauxInfraction.excesVitesse,
                        child: Text('Excès de vitesse (dépend VMA)'),
                      ),
                      DropdownMenuItem(
                        value: EauxInfraction.remousGeneral,
                        child: Text('Remous / succion (contexte)'),
                      ),
                    ],
                    onChanged: (v) {
                      setState(() => _eauxInfraction = v!);
                      _autoQualifyMaybe();
                    },
                  ),
                  if (_eauxInfraction == EauxInfraction.excesVitesse) ...[
                    TextField(
                      controller: _vitesseCtrl,
                      decoration: _dec('Vitesse constatée (km/h)'),
                      keyboardType: const TextInputType.numberWithOptions(
                        decimal: true,
                      ),
                    ),
                    const SizedBox(height: 8),
                    TextField(
                      controller: _vmaCtrl,
                      decoration: _dec('VMA voie navigable (km/h)'),
                      keyboardType: const TextInputType.numberWithOptions(
                        decimal: true,
                      ),
                    ),
                  ] else ...[
                    DropdownButtonFormField<RemousContext>(
                      value: _remousContext,
                      isExpanded: true,
                      decoration: _dec('Contexte remous/succion'),
                      items: const [
                        DropdownMenuItem(
                          value: RemousContext.presBateauAmarre,
                          child: Text('Près d’un bateau amarré (30516)'),
                        ),
                        DropdownMenuItem(
                          value: RemousContext.droitBateauProtege,
                          child: Text('Au droit d’un bateau protégé (30522)'),
                        ),
                        DropdownMenuItem(
                          value: RemousContext.presBac,
                          child: Text('Près d’un bac (30519)'),
                        ),
                        DropdownMenuItem(
                          value: RemousContext.devantEntreePort,
                          child: Text('Devant l’entrée d’un port (30517)'),
                        ),
                        DropdownMenuItem(
                          value: RemousContext.presBateauxStationnement,
                          child: Text(
                            'Près de bateaux en stationnement (30518)',
                          ),
                        ),
                        DropdownMenuItem(
                          value: RemousContext.secteurReglemente,
                          child: Text('Secteur réglementé (30520)'),
                        ),
                        DropdownMenuItem(
                          value: RemousContext.bateauPassagers,
                          child: Text('Bateau à passagers (30455)'),
                        ),
                      ],
                      onChanged: (v) {
                        setState(() => _remousContext = v!);
                        _autoQualifyMaybe();
                      },
                    ),
                  ],
                ],

                // Rhénan (qualification directe)
                if (_domain == Domain.rhenan) ...[
                  const Divider(height: 24),
                  const Text(
                    'Navigation rhénane',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  DropdownButtonFormField<RhenanType>(
                    value: _rhenanType,
                    isExpanded: true,
                    decoration: _dec('Type'),
                    items: const [
                      DropdownMenuItem(
                        value: RhenanType.general,
                        child: Text('Général (33188)'),
                      ),
                      DropdownMenuItem(
                        value: RhenanType.passagers,
                        child: Text('Bâtiment à passagers (33189)'),
                      ),
                      DropdownMenuItem(
                        value: RhenanType.matDangereuses,
                        child: Text(
                          'Transport de matières dangereuses (33190)',
                        ),
                      ),
                      DropdownMenuItem(
                        value: RhenanType.proximiteMatDang,
                        child: Text('À proximité d’un bâtiment MD (33191)'),
                      ),
                    ],
                    onChanged: (v) {
                      setState(() => _rhenanType = v!);
                      _autoQualifyMaybe();
                    },
                  ),
                ],

                // Moselle
                if (_domain == Domain.moselle) ...[
                  const Divider(height: 24),
                  const Text(
                    'Moselle internationale : excès de vitesse (33699)',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  TextField(
                    controller: _vitesseCtrl,
                    decoration: _dec('Vitesse constatée (km/h)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  TextField(
                    controller: _vmaCtrl,
                    decoration: _dec('Vitesse maximale autorisée (VMA) (km/h)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                ],

                // Moyenne
                if (_mode == SpeedCalcMode.moyenne) ...[
                  const Divider(height: 24),
                  TextField(
                    controller: _pk1Ctrl,
                    decoration: _dec('PK départ (km)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  TextField(
                    controller: _pk2Ctrl,
                    decoration: _dec('PK arrivée (km)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  ListTile(
                    title: const Text('Heure départ'),
                    subtitle: Text(_tDep?.format(context) ?? 'Sélectionner'),
                    trailing: const Icon(Icons.access_time),
                    onTap: () => _pickTime(true),
                  ),
                  ListTile(
                    title: const Text('Heure arrivée'),
                    subtitle: Text(_tArr?.format(context) ?? 'Sélectionner'),
                    trailing: const Icon(Icons.access_time),
                    onTap: () => _pickTime(false),
                  ),
                  const SizedBox(height: 8),
                  if (_domain == Domain.sansMoteur) ...[
                    TextField(
                      controller: _vmaSansMoteurCtrl,
                      decoration: _dec(
                        'Vitesse maximale autorisée (VMA) – optionnel',
                      ),
                      keyboardType: const TextInputType.numberWithOptions(
                        decimal: true,
                      ),
                    ),
                  ] else ...[
                    TextField(
                      controller: _vmaCtrl,
                      decoration: _dec(
                        'Vitesse maximale autorisée (VMA) – optionnel',
                      ),
                      keyboardType: const TextInputType.numberWithOptions(
                        decimal: true,
                      ),
                    ),
                  ],
                  if (_domain == Domain.route) ...[
                    const SizedBox(height: 8),
                    DropdownButtonFormField<DriverType>(
                      value: _driver,
                      isExpanded: true,
                      decoration: _dec('Type de conducteur'),
                      items: const [
                        DropdownMenuItem(
                          value: DriverType.normal,
                          child: Text('Conducteur normal'),
                        ),
                        DropdownMenuItem(
                          value: DriverType.eleve,
                          child: Text('Élève conducteur'),
                        ),
                      ],
                      onChanged: (v) => setState(() => _driver = v!),
                    ),
                    SwitchListTile(
                      value: _redevable,
                      onChanged: (v) => setState(() => _redevable = v),
                      title: const Text(
                        'Redevable de l’amende (titulaire CG / personne morale)',
                      ),
                    ),
                    SwitchListTile(
                      value: _recidive50,
                      onChanged: (v) => setState(() => _recidive50 = v),
                      title: const Text('Récidive (si dépassement ≥ 50 km/h)'),
                    ),
                    DropdownButtonFormField<AccidentType>(
                      value: _accident,
                      isExpanded: true,
                      decoration: _dec('Accident corporel (optionnel)'),
                      items: const [
                        DropdownMenuItem(
                          value: AccidentType.aucun,
                          child: Text('Aucun'),
                        ),
                        DropdownMenuItem(
                          value: AccidentType.blessuresLe3Mois,
                          child: Text('Blessures ≤ 3 mois'),
                        ),
                        DropdownMenuItem(
                          value: AccidentType.blessuresGt3Mois,
                          child: Text('Blessures > 3 mois'),
                        ),
                        DropdownMenuItem(
                          value: AccidentType.homicide,
                          child: Text('Homicide involontaire'),
                        ),
                      ],
                      onChanged: (v) => setState(() => _accident = v!),
                    ),
                  ],
                ],

                // Isochrone
                if (_mode == SpeedCalcMode.isochrone) ...[
                  const Divider(height: 24),
                  TextField(
                    controller: _pk1Ctrl,
                    decoration: _dec('PK départ (km)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  TextField(
                    controller: _pk2Ctrl,
                    decoration: _dec('PK arrivée (km)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                  const SizedBox(height: 8),
                  TextField(
                    controller: _vitesseCtrl,
                    decoration: _dec('Vitesse (km/h)'),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                  ),
                ],

                // Distance d’arrêt
                if (_mode == SpeedCalcMode.distanceArret) ...[
                  const Divider(height: 24),
                  const Text(
                    'Distance d’arrêt / vitesse',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  RadioListTile<DistanceCalcMode>(
                    title: const Text('Depuis la vitesse'),
                    value: DistanceCalcMode.fromSpeed,
                    groupValue: _distMode,
                    onChanged: (v) => setState(() => _distMode = v!),
                  ),
                  RadioListTile<DistanceCalcMode>(
                    title: const Text('Depuis la distance'),
                    value: DistanceCalcMode.fromDistance,
                    groupValue: _distMode,
                    onChanged: (v) => setState(() => _distMode = v!),
                  ),
                  if (_distMode == DistanceCalcMode.fromSpeed)
                    TextField(
                      controller: _vitesseCtrl,
                      decoration: _dec('Vitesse (km/h)'),
                      keyboardType: const TextInputType.numberWithOptions(
                        decimal: true,
                      ),
                    )
                  else
                    TextField(
                      controller: _distanceCtrl,
                      decoration: _dec('Distance d’arrêt (m)'),
                      keyboardType: const TextInputType.numberWithOptions(
                        decimal: true,
                      ),
                    ),
                  const SizedBox(height: 8),
                  const Text(
                    'Chaussée',
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  RadioListTile<SurfaceCondition>(
                    title: const Text('Sèche'),
                    value: SurfaceCondition.sec,
                    groupValue: _surface,
                    onChanged: (v) => setState(() => _surface = v!),
                  ),
                  RadioListTile<SurfaceCondition>(
                    title: const Text('Mouillée'),
                    value: SurfaceCondition.mouille,
                    groupValue: _surface,
                    onChanged: (v) => setState(() => _surface = v!),
                  ),
                ],

                // Circonstances
                if (_mode == SpeedCalcMode.circonstances) ...[
                  const Divider(height: 24),
                  SwitchListTile(
                    value: _redevable,
                    onChanged: (v) => setState(() => _redevable = v),
                    title: const Text(
                      'Redevable de l’amende (titulaire CG / personne morale)',
                    ),
                  ),
                ],

                // Incitation
                if (_mode == SpeedCalcMode.incitation) ...[
                  const Divider(height: 24),
                  DropdownButtonFormField<IncitationActeur>(
                    value: _incitActeur,
                    isExpanded: true,
                    decoration: _dec('Acteur'),
                    items: const [
                      DropdownMenuItem(
                        value: IncitationActeur.employeur,
                        child: Text('Employeur'),
                      ),
                      DropdownMenuItem(
                        value: IncitationActeur.donneurOrdre,
                        child: Text("Donneur d’ordre"),
                      ),
                    ],
                    onChanged: (v) => setState(() => _incitActeur = v!),
                  ),
                  const SizedBox(height: 8),
                  DropdownButtonFormField<IncitationTransport>(
                    value: _incitTransport,
                    isExpanded: true,
                    decoration: _dec('Type de transport'),
                    items: const [
                      DropdownMenuItem(
                        value: IncitationTransport.marchandises,
                        child: Text('Marchandises'),
                      ),
                      DropdownMenuItem(
                        value: IncitationTransport.personnes,
                        child: Text('Personnes'),
                      ),
                      DropdownMenuItem(
                        value: IncitationTransport.matDangOuExceptionnel,
                        child: Text('Matières dangereuses / exceptionnel'),
                      ),
                    ],
                    onChanged: (v) => setState(() => _incitTransport = v!),
                  ),
                ],

                const SizedBox(height: 12),
                if (showAction)
                  ElevatedButton.icon(
                    onPressed: canCalc ? _calculate : null,
                    icon: const Icon(Icons.calculate),
                    label: const Text('Calculer'),
                  ),

                const SizedBox(height: 20),

                if (_error != null)
                  Text(_error!, style: const TextStyle(color: Colors.red)),

                if (_resultat != null) ...[
                  Text(
                    _resultat!,
                    style: const TextStyle(
                      fontSize: 18,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  if (_explication != null) ...[
                    const SizedBox(height: 6),
                    Text(
                      _explication!,
                      style: const TextStyle(fontStyle: FontStyle.italic),
                    ),
                  ],
                  if (_classification != null) ...[
                    const SizedBox(height: 6),
                    Text('Classification : $_classification'),
                  ],
                  const SizedBox(height: 12),
                  if (_loading)
                    const Center(child: CircularProgressIndicator())
                  else if (_natinfResults != null &&
                      _natinfResults!.isNotEmpty) ...[
                    Text(
                      _natinfResults!.length > 1
                          ? 'Code(s) NATINF applicable(s) :'
                          : 'Code NATINF applicable :',
                      style: const TextStyle(
                        fontSize: 16,
                        fontWeight: FontWeight.bold,
                      ),
                    ),
                    const SizedBox(height: 8),
                    NatinfList(
                      natinfList: _natinfResults!,
                      recordHistory: false,
                      nested: true,
                    ),
                  ] else if (_natinfCodes != null &&
                      _natinfCodes!.isNotEmpty) ...[
                    Text('Code(s) NATINF : ${_natinfCodes!.join(', ')}'),
                  ],
                  if ((_natinfCodes != null && _natinfCodes!.isNotEmpty) ||
                      (_natinfResults != null &&
                          _natinfResults!.isNotEmpty)) ...[
                    const SizedBox(height: 12),
                    _disclaimerCard(),
                  ],
                  const SizedBox(height: 8),
                  const Divider(height: 32),
                ],
              ],
            ),
          );
        },
      ),
      bottomNavigationBar:
          (_mode == SpeedCalcMode.cinemometre && _needsSpeedFields()) ||
                  _mode == SpeedCalcMode.distanceArret
              ? Padding(
                padding: const EdgeInsets.all(16),
                child: Column(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    if (_mode == SpeedCalcMode.cinemometre &&
                        _needsSpeedFields())
                      TextButton(
                        onPressed: () async {
                          final ok = await launchUrl(Uri.parse(_urlCine));
                          if (!ok && mounted) {
                            ScaffoldMessenger.of(context).showSnackBar(
                              const SnackBar(
                                content: Text('Impossible d’ouvrir le lien'),
                              ),
                            );
                          }
                        },
                        child: const Text(
                          'Arrêté du 4 juin 2009 relatif aux cinémomètres – Légifrance',
                          textAlign: TextAlign.center,
                          style: TextStyle(
                            decoration: TextDecoration.underline,
                          ),
                        ),
                      ),
                    if (_mode == SpeedCalcMode.distanceArret)
                      TextButton(
                        onPressed: () async {
                          final ok = await launchUrl(
                            Uri.parse(_urlDistanceArret),
                          );
                          if (!ok && mounted) {
                            ScaffoldMessenger.of(context).showSnackBar(
                              const SnackBar(
                                content: Text('Impossible d’ouvrir le lien'),
                              ),
                            );
                          }
                        },
                        child: const Text(
                          'Vitesse & distance d’arrêt – Ministère de l’Éducation nationale',
                          textAlign: TextAlign.center,
                          style: TextStyle(
                            decoration: TextDecoration.underline,
                          ),
                        ),
                      ),
                  ],
                ),
              )
              : null,
    );
  }
}
