import 'dart:math' as math;

import 'package:flutter/material.dart';
import 'package:meta/meta.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_details_page.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:provider/provider.dart';

class NatinfList extends StatefulWidget {
  final List<Natinf> natinfList;
  final bool recordHistory;
  final bool nested;

  /// Callback triggered when a NATINF card is tapped. Defaults to opening the
  /// NATINF detail page.
  final Future<void> Function(BuildContext context, Natinf natinf)? onNatinfTap;

  /// Builder used to display a trailing widget for each NATINF entry.
  final Widget Function(BuildContext context, Natinf natinf)? trailingBuilder;

  const NatinfList({
    super.key,
    required this.natinfList,
    this.recordHistory = true,
    this.nested = false,
    this.onNatinfTap,
    this.trailingBuilder,
  });

  @override
  State<NatinfList> createState() => _NatinfListState();
}

class _NatinfListState extends State<NatinfList>
    with SingleTickerProviderStateMixin {
  static const _baseSlideOffset = Offset(0, 0.08);
  static const _animationDuration = Duration(milliseconds: 600);

  late final AnimationController _controller;

  @override
  void initState() {
    super.initState();
    _controller = AnimationController(duration: _animationDuration, vsync: this)
      ..forward();
  }

  @override
  void didUpdateWidget(covariant NatinfList oldWidget) {
    super.didUpdateWidget(oldWidget);
    final listChanged =
        !_hasSameNatinfs(widget.natinfList, oldWidget.natinfList);
    if (listChanged) {
      _controller.forward(from: 0);
    }
  }

  @override
  void dispose() {
    _controller.dispose();
    super.dispose();
  }

  @visibleForTesting
  double get animationProgress => _controller.value;

  @override
  Widget build(BuildContext context) {
    return ListView.builder(
      shrinkWrap: widget.nested,
      physics: widget.nested ? const NeverScrollableScrollPhysics() : null,
      itemCount: widget.natinfList.length,
      itemBuilder: (context, index) {
        final natinf = widget.natinfList[index];
        final animation = CurvedAnimation(
          parent: _controller,
          curve: Interval(
            _staggerStart(index, widget.natinfList.length),
            _staggerEnd(index, widget.natinfList.length),
            curve: Curves.easeOutCubic,
          ),
        );

        return FadeTransition(
          key: ValueKey(natinf.numeroNatinf ?? index),
          opacity: animation,
          child: SlideTransition(
            position: Tween<Offset>(
              begin: _baseSlideOffset,
              end: Offset.zero,
            ).animate(animation),
            child: _NatinfCard(
              natinf: natinf,
              recordHistory: widget.recordHistory,
              trailingBuilder: widget.trailingBuilder,
              onTap: widget.onNatinfTap,
            ),
          ),
        );
      },
    );
  }

  double _staggerStart(int index, int length) {
    if (length <= 1) {
      return 0;
    }
    final fraction = index / length;
    return math.max(0, fraction * 0.9);
  }

  double _staggerEnd(int index, int length) {
    final start = _staggerStart(index, length);
    return math.min(1, start + 0.3);
  }

  bool _hasSameNatinfs(List<Natinf> current, List<Natinf> previous) {
    if (identical(current, previous)) {
      return true;
    }
    if (current.length != previous.length) {
      return false;
    }
    for (var i = 0; i < current.length; i++) {
      final currentKey =
          current[i].numeroNatinf ?? current[i].id?.toString() ?? '';
      final previousKey =
          previous[i].numeroNatinf ?? previous[i].id?.toString() ?? '';
      if (currentKey != previousKey) {
        return false;
      }
    }
    return true;
  }
}

class _NatinfCard extends StatelessWidget {
  final Natinf natinf;
  final bool recordHistory;
  final Widget Function(BuildContext context, Natinf natinf)? trailingBuilder;
  final Future<void> Function(BuildContext context, Natinf natinf)? onTap;

  const _NatinfCard({
    required this.natinf,
    required this.recordHistory,
    this.trailingBuilder,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final theme = Provider.of<ThemeProvider>(context, listen: false);
    final repository = context.read<NatinfRepository>();
    final indicatorColor =
        natinf.natureInfraction == null
            ? Colors.transparent
            : theme.colorForInfractionLabel(natinf.natureInfraction!);
    final bool isObsolete = natinf.obsoleteDate != null;

    return Card(
      margin: const EdgeInsets.symmetric(horizontal: 4, vertical: 8),
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
      elevation: 2,
      color: isObsolete ? Colors.red.withOpacity(0.2) : null,
      child: InkWell(
        borderRadius: BorderRadius.circular(12),
        onTap: () async {
          FocusScope.of(context).unfocus();
          if (recordHistory && natinf.numeroNatinf != null) {
            await context.read<NatinfRepository>().recordHistoryEntry(
              natinf.numeroNatinf!,
            );
          }
          if (onTap != null) {
            await onTap!(context, natinf);
          } else {
            await Navigator.push(
              context,
              MaterialPageRoute(builder: (_) => NatinfDetail(natinf: natinf)),
            );
          }
        },
        child: IntrinsicHeight(
          child: Row(
            crossAxisAlignment: CrossAxisAlignment.stretch,
            children: [
              // bandeau coloré qui remplit toute la hauteur
              Container(
                width: 6,
                decoration: BoxDecoration(
                  color: indicatorColor,
                  borderRadius: const BorderRadius.only(
                    topLeft: Radius.circular(12),
                    bottomLeft: Radius.circular(12),
                  ),
                ),
              ),

              // contenu principal
              Expanded(
                child: Padding(
                  padding: const EdgeInsets.symmetric(
                    horizontal: 16,
                    vertical: 12,
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      // 1) Qualification en gros
                      if (natinf.qualificationInfraction != null)
                        Text(
                          natinf.qualificationInfraction!,
                          style: const TextStyle(
                            fontSize: 14,
                            fontWeight: FontWeight.bold,
                            height: 1.2,
                          ),
                        ),

                      const SizedBox(height: 8),

                      // 2) Numéro NATINF
                      if (natinf.numeroNatinf != null)
                        Text(
                          'NATINF ${natinf.numeroNatinf}',
                          style: const TextStyle(fontSize: 14),
                        ),

                      // 3) Nature de l’infraction
                      if (natinf.natureInfraction != null)
                        Padding(
                          padding: const EdgeInsets.only(top: 4),
                          child: Text(
                            natinf.natureInfraction!,
                            style: const TextStyle(fontSize: 14),
                          ),
                        ),
                    ],
                  ),
                ),
              ),

              // icône de navigation
              Padding(
                padding: const EdgeInsets.only(right: 16),
                child:
                    trailingBuilder?.call(context, natinf) ??
                    const Icon(Icons.chevron_right, size: 20),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
