import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';

typedef LocalSearchInvoker = Future<List<Natinf>> Function(String query);
typedef ConnectivityLookup = Future<ConnectivityResult> Function();

/// Describes why the repository should skip the remote search.
enum OfflineMode { none, forced, noConnectivity }

/// Determines whether the search must be executed locally before reaching the
/// network. The `connectivityLookup` parameter makes the logic testable by
/// letting callers inject a fake resolver.
Future<OfflineMode> resolveOfflineMode({
  required bool forceOffline,
  ConnectivityLookup? connectivityLookup,
}) async {
  if (forceOffline) {
    return OfflineMode.forced;
  }
  final lookup = connectivityLookup ?? Connectivity().checkConnectivity;
  final connectivity = await lookup();
  if (connectivity == ConnectivityResult.none) {
    return OfflineMode.noConnectivity;
  }
  return OfflineMode.none;
}

/// Runs a local database search and reports a status message only when
/// results are available. The caller can decide how to handle an empty list.
Future<List<Natinf>> fallbackToLocalSearch(
  String query, {
  required LocalSearchInvoker localSearch,
  void Function(String)? onStatus,
  required String successStatus,
}) async {
  final results = await localSearch(query);
  if (results.isNotEmpty) {
    onStatus?.call(successStatus);
  }
  return results;
}
