import 'package:flutter/material.dart';
import 'package:provider/provider.dart';

import '../presentation/auth_provider.dart';

/// Basic login screen for NATINFo authentication.
class LoginPage extends StatefulWidget {
  const LoginPage({super.key});

  @override
  State<LoginPage> createState() => _LoginPageState();
}

class _LoginPageState extends State<LoginPage> {
  static const String _defaultTokenLabel = 'NATINFo+';
  final _formKey = GlobalKey<FormState>();
  final _usernameController = TextEditingController();
  final _passwordController = TextEditingController();
  final _tokenLabelController = TextEditingController(text: _defaultTokenLabel);
  bool _submitting = false;
  String? _error;

  @override
  void dispose() {
    _usernameController.dispose();
    _passwordController.dispose();
    _tokenLabelController.dispose();
    super.dispose();
  }

  Future<void> _submit() async {
    if (!_formKey.currentState!.validate()) {
      return;
    }
    setState(() {
      _submitting = true;
      _error = null;
    });
    final auth = context.read<AuthProvider>();
    final rawTokenLabel = _tokenLabelController.text.trim();
    final tokenLabel =
        rawTokenLabel.isEmpty ? _defaultTokenLabel : rawTokenLabel;
    try {
      await auth.login(
        username: _usernameController.text.trim(),
        password: _passwordController.text,
        tokenLabel: tokenLabel,
      );
      if (!mounted) return;
      Navigator.of(context).pop(true);
      ScaffoldMessenger.of(
        context,
      ).showSnackBar(const SnackBar(content: Text('Connexion réussie')));
    } catch (error) {
      setState(() {
        _error = 'Échec de connexion (${error.runtimeType})';
      });
    } finally {
      if (mounted) {
        setState(() {
          _submitting = false;
        });
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    final padding = MediaQuery.of(context).padding;
    return Scaffold(
      appBar: AppBar(title: const Text('Connexion')),
      body: Padding(
        padding: EdgeInsets.fromLTRB(
          16,
          padding.top + 16,
          16,
          padding.bottom + 16,
        ),
        child: Form(
          key: _formKey,
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.stretch,
            children: [
              TextFormField(
                controller: _usernameController,
                decoration: const InputDecoration(
                  labelText: 'Nom d’utilisateur',
                ),
                textInputAction: TextInputAction.next,
                validator:
                    (value) =>
                        (value == null || value.isEmpty) ? 'Requis' : null,
              ),
              const SizedBox(height: 12),
              TextFormField(
                controller: _passwordController,
                decoration: const InputDecoration(labelText: 'Mot de passe'),
                obscureText: true,
                validator:
                    (value) =>
                        (value == null || value.isEmpty) ? 'Requis' : null,
              ),
              const SizedBox(height: 12),
              TextFormField(
                controller: _tokenLabelController,
                decoration: const InputDecoration(
                  labelText: 'Libellé du jeton (optionnel)',
                ),
              ),
              const SizedBox(height: 16),
              if (_error != null)
                Padding(
                  padding: const EdgeInsets.only(bottom: 8),
                  child: Text(
                    _error!,
                    style: const TextStyle(color: Colors.red),
                  ),
                ),
              ElevatedButton(
                onPressed: _submitting ? null : _submit,
                child:
                    _submitting
                        ? const SizedBox(
                          height: 20,
                          width: 20,
                          child: CircularProgressIndicator(strokeWidth: 2),
                        )
                        : const Text('Se connecter'),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
