import 'package:flutter/foundation.dart' hide Category;
import 'package:intl/intl.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:shared_preferences/shared_preferences.dart';

/// Bundles the dependencies required to start the application.
class RepositoryBootstrap {
  final SharedPreferences preferences;
  final ThemeProvider themeProvider;
  final AuthRepository authRepository;
  final AuthProvider authProvider;
  final NatinfRepository repository;

  RepositoryBootstrap({
    required this.preferences,
    required this.themeProvider,
    required this.authRepository,
    required this.authProvider,
    required this.repository,
  });

  /// Initializes preferences, theme provider, and the NATINF repository.
  ///
  /// On failure, a minimal fallback is returned to let the app start with
  /// reduced functionality instead of crashing at cold start.
  static Future<RepositoryBootstrap> initialize({
    Future<SharedPreferences> Function()? preferencesLoader,
    NatinfRepository Function()? repositoryBuilder,
    AuthRepository Function()? authRepositoryBuilder,
    AuthProvider Function(AuthRepository repository)? authProviderBuilder,
    Uri? authBaseUri,
    void Function(String message)? log,
  }) async {
    final logger = log ?? debugPrint;
    Intl.defaultLocale = 'fr_FR';

    final prefs = await _loadPreferences(
      loader: preferencesLoader,
      log: logger,
    );
    AppConfig.forceOffline = prefs.getBool('forceOffline') ?? false;

    final themeProvider = ThemeProvider();
    await themeProvider.load(prefs: prefs, notify: false);

    final resolvedAuthRepository =
        authRepositoryBuilder?.call() ??
        AuthRepository(
          baseUri: authBaseUri ?? Uri.parse('https://natinfo.app/api'),
          storage: SecureAuthStorage(sharedPreferences: prefs),
        );
    final authProvider =
        authProviderBuilder?.call(resolvedAuthRepository) ??
        AuthProvider(resolvedAuthRepository);
    await authProvider.bootstrap(forceOffline: AppConfig.forceOffline);

    final repository = await _buildRepository(
      builder:
          repositoryBuilder ??
          () => NatinfRepository(
            authRepository: resolvedAuthRepository,
            authBaseUri: authBaseUri,
          ),
      log: logger,
    );

    return RepositoryBootstrap(
      preferences: prefs,
      themeProvider: themeProvider,
      authRepository: resolvedAuthRepository,
      authProvider: authProvider,
      repository: repository,
    );
  }

  static Future<SharedPreferences> _loadPreferences({
    Future<SharedPreferences> Function()? loader,
    void Function(String message)? log,
  }) async {
    try {
      return await (loader ?? SharedPreferences.getInstance)();
    } catch (error, stackTrace) {
      log?.call(
        'SharedPreferences init failed, falling back to in-memory store: '
        '$error\n$stackTrace',
      );
      SharedPreferences.setMockInitialValues(const {});
      return SharedPreferences.getInstance();
    }
  }

  static Future<NatinfRepository> _buildRepository({
    required NatinfRepository Function() builder,
    void Function(String message)? log,
  }) async {
    try {
      final repository = builder();
      await repository.init();
      return repository;
    } catch (error, stackTrace) {
      log?.call(
        'NATINF repository init failed, continuing with limited mode: '
        '$error\n$stackTrace',
      );
      return _FallbackNatinfRepository(error);
    }
  }
}

/// Signals that the repository could not be initialized and is running in a
/// degraded mode.
class RepositoryUnavailableException implements Exception {
  RepositoryUnavailableException(this.message, {this.cause});

  /// User-facing description of the unavailability reason.
  final String message;

  /// Optional underlying cause.
  final Object? cause;

  @override
  String toString() =>
      'RepositoryUnavailableException: $message'
      '${cause != null ? ' (cause: $cause)' : ''}';
}

class _FallbackNatinfRepository extends NatinfRepository {
  _FallbackNatinfRepository(this._reason);

  final Object _reason;

  Future<T> _unavailable<T>() {
    return Future.error(
      RepositoryUnavailableException(
        "Le dépôt NATINF n'a pas pu démarrer (mode dégradé). "
        "Fonctionnalités limitées.",
        cause: _reason,
      ),
    );
  }

  @override
  Future<void> init() async {
    // No-op: fallback repository starts without external state.
  }

  @override
  Future<bool> hasLocalData() async => false;

  @override
  bool isDocsProUnauthorized(String numero) => false;

  @override
  Future<List<CategoryBrief>> getCachedRootCategories() async => const [];

  @override
  Future<Category?> getCachedCategory(int id) async => null;

  @override
  Future<Natinf?> getNatinfByNumero(String numero) async => null;

  @override
  Future<void> recordHistoryEntry(String numero) async {}

  @override
  Future<List<CategoryBrief>> fetchRootCategories() async => const [];

  @override
  Future<Category> fetchCategory(int id) => _unavailable();

  @override
  Future<void> fetchInitialData({
    void Function(String p1)? onStatus,
    void Function(double p1)? onProgress,
    void Function(int p1)? onPhase,
  }) => _unavailable();

  @override
  Future<List<Natinf>> searchNatinf(
    String query, {
    void Function(String p1)? onStatus,
    bool forceOffline = false,
  }) {
    onStatus?.call(
      "Données NATINF indisponibles (mode dégradé après échec d'initialisation).",
    );
    return _unavailable();
  }
}
