import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/auth/presentation/login_page.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/favourites_page.dart';
import 'package:provider/provider.dart';
import 'package:sembast/sembast_io.dart';

void main() {
  late bool previousForceOffline;
  late Directory tempDir;
  late AuthProvider signedOutAuth;
  late AuthRepository authRepository;

  setUp(() async {
    previousForceOffline = AppConfig.forceOffline;
    AppConfig.forceOffline = true;
    tempDir = await Directory.systemTemp.createTemp('fav-page-test');
    await DatabaseHelper().initDbForTesting(
      databaseFactoryIo,
      '${tempDir.path}/db.db',
    );
    authRepository = _buildAuthRepository();
    signedOutAuth = AuthProvider(authRepository);
    await signedOutAuth.bootstrap();
  });

  tearDown(() async {
    signedOutAuth.dispose();
    await tempDir.delete(recursive: true);
    AppConfig.forceOffline = previousForceOffline;
  });

  testWidgets('shows sync menu even when signed out', (tester) async {
    await tester.pumpWidget(
      _buildHost(authProvider: signedOutAuth, authRepository: authRepository),
    );

    await tester.pumpAndSettle();

    expect(find.byTooltip('Synchroniser les favoris'), findsOneWidget);
  });

  testWidgets(
    'tapping sync while signed out proposes login and opens login page',
    (tester) async {
      await tester.pumpWidget(
        _buildHost(authProvider: signedOutAuth, authRepository: authRepository),
      );

      await tester.pumpAndSettle();

      await tester.tap(find.byTooltip('Synchroniser les favoris'));
      await tester.pumpAndSettle();

      await tester.tap(find.text('Sauvegarder mes favoris'));
      await tester.pumpAndSettle();

      expect(find.text('Connexion requise'), findsOneWidget);
      await tester.tap(find.text('Se connecter'));
      await tester.pumpAndSettle();

      expect(find.byType(LoginPage), findsOneWidget);
    },
  );
}

Widget _buildHost({
  required AuthProvider authProvider,
  required AuthRepository authRepository,
}) {
  return MultiProvider(
    providers: [
      ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
      Provider<AuthRepository>.value(value: authRepository),
    ],
    child: const MaterialApp(home: FavouritesPage()),
  );
}

AuthRepository _buildAuthRepository() {
  final storage = InMemoryAuthStorage();
  return AuthRepository(
    baseUri: Uri.parse('https://natinfo.app/api'),
    storage: storage,
    httpClient: MockClient((request) async => http.Response('{}', 200)),
  );
}
