import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/domain/auth_session.dart';
import 'package:natinfo_flutter/features/auth/presentation/account_page.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:provider/provider.dart';

void main() {
  group('AccountPage layout', () {
    late InMemoryAuthStorage storage;
    late AuthRepository repository;

    setUp(() {
      storage = InMemoryAuthStorage();
      repository = AuthRepository(
        baseUri: Uri.parse('https://natinfo.app/api'),
        storage: storage,
        httpClient: MockClient((_) async => http.Response('{}', 200)),
      );
    });

    testWidgets('shows organized sections without exposing user ID', (
      tester,
    ) async {
      final session = AuthSession(
        accessToken: 'access',
        refreshToken: 'refresh',
        accessExpiresAt: DateTime(2030, 1, 1, 12).toUtc(),
        refreshExpiresAt: DateTime(2030, 2, 1, 12).toUtc(),
        userId: 42,
        username: 'alice',
        hasDocPro: true,
        lastRefresh: DateTime(2025, 1, 2, 15, 30).toUtc(),
        tokenLabel: 'Jeton principal',
      );
      await storage.save(session);
      final provider = AuthProvider(repository);
      await provider.bootstrap();

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider<AuthProvider>.value(value: provider),
          ],
          child: const MaterialApp(home: AccountPage()),
        ),
      );
      await tester.pumpAndSettle();

      expect(find.text('Utilisateur'), findsOneWidget);
      expect(find.text('alice'), findsOneWidget);
      expect(find.textContaining('ID'), findsNothing);

      expect(find.text('DocPro'), findsOneWidget);
      expect(find.text('Statut des jetons'), findsOneWidget);
      expect(find.text('Expiration refresh'), findsOneWidget);
      expect(find.text('Dernier refresh'), findsOneWidget);

      final docProTop = tester.getTopLeft(find.text('DocPro')).dy;
      final tokensTop = tester.getTopLeft(find.text('Statut des jetons')).dy;
      expect(docProTop, lessThan(tokensTop));
    });

    testWidgets(
      'renders token timing group with fallback when never refreshed',
      (tester) async {
        final session = AuthSession(
          accessToken: 'access',
          refreshToken: 'refresh',
          accessExpiresAt: DateTime(2030, 3, 10, 12).toUtc(),
          refreshExpiresAt: DateTime(2030, 4, 10, 12).toUtc(),
          userId: 99,
          username: 'bob',
          hasDocPro: false,
        );
        await storage.save(session);
        final provider = AuthProvider(repository);
        await provider.bootstrap();

        await tester.pumpWidget(
          MultiProvider(
            providers: [
              ChangeNotifierProvider<AuthProvider>.value(value: provider),
            ],
            child: const MaterialApp(home: AccountPage()),
          ),
        );
        await tester.pumpAndSettle();

        expect(find.text('Non rafraîchi'), findsOneWidget);
        expect(find.text('Inactif'), findsOneWidget);
        expect(find.text('DocPro'), findsOneWidget);
      },
    );
  });
}
