import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';
import 'package:natinfo_flutter/shared/utils/iterable_extensions.dart';

/// Keeps a catalog of declared sources and provides filtered views on them.
class SourceRegistry {
  SourceRegistry(List<SourceSpec> sources)
    : this._(List<SourceSpec>.from(sources)..sort(_priorityComparator));

  SourceRegistry._(List<SourceSpec> sortedSources)
    : _sources = List<SourceSpec>.unmodifiable(sortedSources),
      _datasetNames = _buildDatasetNames(sortedSources) {
    final uniqueIds = <String>{};
    for (final source in _sources) {
      if (!uniqueIds.add(source.id)) {
        throw ArgumentError.value(
          source.id,
          'sources',
          'Duplicate source id detected',
        );
      }
    }
  }

  /// Builds a registry from the JSON string bundled in assets/source_registry.json.
  ///
  /// When [overrides] are provided, they inject user-defined sources for the
  /// corresponding datasets.
  factory SourceRegistry.fromJsonString(
    String jsonString, {
    Map<String, Uri>? overrides,
  }) {
    final specs = SourceSpec.listFromJsonString(jsonString);
    final merged =
        overrides == null || overrides.isEmpty
            ? specs
            : _applyOverrides(specs, overrides);
    return SourceRegistry(merged);
  }

  final List<SourceSpec> _sources;
  final Map<String, String> _datasetNames;

  /// Returns all sources in deterministic priority order.
  List<SourceSpec> get all => _sources;

  /// Human-readable dataset names keyed by dataset id.
  Map<String, String> get datasetNames => _datasetNames;

  /// Human-readable name for [dataset], falling back to the dataset id.
  String nameForDataset(String dataset) {
    return _datasetNames[dataset] ?? dataset;
  }

  /// Finds a source by its id.
  SourceSpec? findById(String id) {
    return _sources.where((s) => s.id == id).cast<SourceSpec?>().firstOrNull;
  }

  /// Lists sources for the given dataset, sorted by priority.
  ///
  /// If [allowNetwork] is false, sources requiring network are excluded.
  List<SourceSpec> sourcesForDataset(
    String dataset, {
    bool allowNetwork = true,
  }) {
    final filtered = _sources.where((source) => source.dataset == dataset);
    final networkFiltered =
        allowNetwork
            ? filtered
            : filtered.where((source) => !source.requiresNetwork);
    return List<SourceSpec>.unmodifiable(networkFiltered);
  }

  static int _priorityComparator(SourceSpec a, SourceSpec b) {
    final priority = a.priority.compareTo(b.priority);
    if (priority != 0) return priority;
    return a.id.compareTo(b.id);
  }

  static List<SourceSpec> _applyOverrides(
    List<SourceSpec> base,
    Map<String, Uri> overrides,
  ) {
    if (overrides.isEmpty) return base;

    final merged = List<SourceSpec>.from(base);
    final templates = <String, SourceSpec>{};
    for (final spec in base) {
      templates.putIfAbsent(spec.dataset, () => spec);
    }

    overrides.forEach((dataset, uri) {
      final template = templates[dataset];
      if (template == null) return;
      if (!template.userEditable) return;
      final requiresNetwork =
          template.requiresNetwork ||
          uri.scheme == 'http' ||
          uri.scheme == 'https';
      merged.add(
        SourceSpec(
          id: 'custom-$dataset',
          dataset: template.dataset,
          name: template.name,
          license: template.license,
          type: template.type,
          uri: uri,
          scope: template.scope,
          priority: 1,
          requiredAtBuild: template.requiredAtBuild,
          requiresNetwork: requiresNetwork,
          schemaVersion: template.schemaVersion,
          checksum: null,
          checksumAlgo: null,
          userEditable: template.userEditable,
        ),
      );
    });
    return merged;
  }

  static Map<String, String> _buildDatasetNames(List<SourceSpec> sources) {
    final names = <String, String>{};
    for (final spec in sources) {
      final existing = names[spec.dataset];
      if (existing != null && existing != spec.name) {
        throw ArgumentError.value(
          spec.name,
          'name',
          'Conflicting dataset names for "${spec.dataset}"',
        );
      }
      names[spec.dataset] = spec.name;
    }
    return Map<String, String>.unmodifiable(names);
  }
}
