import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_errors.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

/// Contract for loading dataset payloads from configured sources.
abstract class SourceLoadClient {
  Future<SourceLoadResult> load(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
    int? expectedSchemaVersion,
  });
}

/// Loads datasets using the configured sources with automatic fallback.
class SourceLoader implements SourceLoadClient {
  SourceLoader({
    required this.resolver,
    required List<SourceAdapter> adapters,
    SourceLogger? logger,
  }) : _adapters = {for (final adapter in adapters) adapter.type: adapter},
       _logger = logger ?? SourceLogger.shared;

  /// Resolves available sources.
  final SourceResolver resolver;

  /// Registered adapters keyed by source type.
  final Map<SourceType, SourceAdapter> _adapters;

  final SourceLogger _logger;

  /// Loads the dataset, trying each source in priority order until one succeeds.
  ///
  /// On success, returns the [SourceLoadResult] of the source that loaded the
  /// payload. If all sources fail, a [SourceUnavailableException] is thrown.
  @override
  Future<SourceLoadResult> load(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
    int? expectedSchemaVersion,
  }) async {
    final candidates = resolver.resolveCandidates(
      dataset,
      allowNetwork: allowNetwork,
      preferredSourceId: preferredSourceId,
    );
    if (candidates.isEmpty) {
      throw resolver.buildUnavailableException(
        dataset,
        allowNetwork: allowNetwork,
      );
    }
    _logger.info(
      'Resolved ${candidates.length} candidate source(s) for dataset "$dataset".',
    );

    SourceLoadException? lastError;
    for (final spec in candidates) {
      _logger.info(
        'Trying source "${spec.id}" (${spec.type.name}) for dataset "$dataset"...',
      );
      final adapter = _adapters[spec.type];
      if (adapter == null) {
        lastError = SourceLoadException(
          spec,
          UnsupportedError('No adapter registered for ${spec.type}'),
        );
        _logger.warn(
          'Skipping source "${spec.id}" due to missing adapter for ${spec.type}.',
        );
        continue;
      }
      try {
        final result = await adapter.load(spec);
        _handleIntegrity(result);
        _handleSchema(spec, expectedSchemaVersion: expectedSchemaVersion);
        _logger.info(
          'Loaded dataset "$dataset" from source "${spec.id}" (${spec.type.name}).',
        );
        return result;
      } catch (error, stackTrace) {
        lastError = SourceLoadException(spec, error, stackTrace);
        _logger.warn(
          'Source "${spec.id}" failed for dataset "$dataset": $error',
        );
      }
    }

    final requiredIds =
        candidates
            .where((spec) => spec.requiredAtBuild)
            .map((spec) => spec.id)
            .toList();
    final buffer = StringBuffer(
      'Failed to load dataset "$dataset" from configured sources.',
    );
    if (lastError != null) {
      buffer.write(' Last error: ${lastError.error}.');
    }
    throw SourceUnavailableException(
      buffer.toString(),
      dataset: dataset,
      requiredSourceIds: requiredIds,
    );
  }

  void _handleIntegrity(SourceLoadResult result) {
    final integrity = result.integrity;
    switch (integrity.status) {
      case ChecksumStatus.mismatch:
        _logger.warn(
          'Checksum mismatch for source "${result.spec.id}" '
          '(expected ${integrity.expected}, actual ${integrity.actual}).',
        );
        break;
      case ChecksumStatus.skippedUnsupportedAlgorithm:
        _logger.warn(
          'Checksum algorithm "${integrity.algorithm}" not supported for '
          'source "${result.spec.id}".',
        );
        break;
      case ChecksumStatus.notProvided:
      case ChecksumStatus.match:
        break;
    }
  }

  void _handleSchema(SourceSpec spec, {int? expectedSchemaVersion}) {
    if (expectedSchemaVersion == null) return;
    if (spec.schemaVersion != expectedSchemaVersion) {
      _logger.warn(
        'Schema version mismatch for source "${spec.id}" '
        '(expected $expectedSchemaVersion, got ${spec.schemaVersion}).',
      );
    }
  }
}
