import 'dart:io';

import 'package:flutter/services.dart';
import 'package:shared_preferences/shared_preferences.dart';

/// Immutable description of a custom app icon made available to users.
class AppIconOption {
  const AppIconOption({
    required this.key,
    required this.label,
    required this.previewAssetPath,
    this.description,
  });

  /// Unique key used to identify the icon across platforms.
  final String key;

  /// User facing label rendered in the icon picker.
  final String label;

  /// Asset pointing to a 512x512 PNG preview of the icon.
  final String previewAssetPath;

  /// Optional short description displayed under the icon preview.
  final String? description;
}

/// Handles persistence and platform communication for custom app icons.
class AppIconService {
  AppIconService._();

  factory AppIconService() => _instance;

  static const String defaultKey = 'classic';
  static const String _methodChannelName = 'net.retiolus.natinfo/app_icon';
  static const String _preferenceKey = 'selectedAppIconKey';
  static final AppIconService _instance = AppIconService._();
  static const MethodChannel _channel = MethodChannel(_methodChannelName);

  static const List<AppIconOption> options = [
    AppIconOption(
      key: defaultKey,
      label: 'Classique',
      previewAssetPath: 'assets/icon/icon.png',
      description: 'Icône par défaut',
    ),
    AppIconOption(
      key: 'tricolore_fr',
      label: 'Bleu blanc rouge',
      previewAssetPath:
          'assets/app_icon_previews/NATINFoPlus_Tricolore_FR_512.png',
    ),
    AppIconOption(
      key: 'blue',
      label: 'Bleu',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Blue_512.png',
    ),
    AppIconOption(
      key: 'graphite',
      label: 'Graphite',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Graphite_512.png',
    ),
    AppIconOption(
      key: 'green',
      label: 'Vert',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Green_512.png',
    ),
    AppIconOption(
      key: 'indigo',
      label: 'Indigo',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Indigo_512.png',
    ),
    AppIconOption(
      key: 'lime',
      label: 'Citron vert',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Lime_512.png',
    ),
    AppIconOption(
      key: 'navy',
      label: 'Bleu marine',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Navy_512.png',
    ),
    AppIconOption(
      key: 'orange',
      label: 'Orange',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Orange_512.png',
    ),
    AppIconOption(
      key: 'pink',
      label: 'Rose',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Pink_512.png',
    ),
    AppIconOption(
      key: 'purple',
      label: 'Violet',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Purple_512.png',
    ),
    AppIconOption(
      key: 'red',
      label: 'Rouge',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Red_512.png',
    ),
    AppIconOption(
      key: 'teal',
      label: 'Turquoise',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Teal_512.png',
    ),
    AppIconOption(
      key: 'yellow',
      label: 'Jaune',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Yellow_512.png',
    ),
    AppIconOption(
      key: 'pride_ace',
      label: 'Pride asexuel·le',
      previewAssetPath:
          'assets/app_icon_previews/NATINFoPlus_Pride_Ace_512.png',
    ),
    AppIconOption(
      key: 'pride_bi',
      label: 'Pride bi',
      previewAssetPath: 'assets/app_icon_previews/NATINFoPlus_Pride_Bi_512.png',
    ),
    AppIconOption(
      key: 'pride_pan',
      label: 'Pride pan',
      previewAssetPath:
          'assets/app_icon_previews/NATINFoPlus_Pride_Pan_512.png',
    ),
    AppIconOption(
      key: 'pride_progress',
      label: 'Pride progress',
      previewAssetPath:
          'assets/app_icon_previews/NATINFoPlus_Pride_Progress_512.png',
    ),
    AppIconOption(
      key: 'pride_rainbow',
      label: 'Pride arc-en-ciel',
      previewAssetPath:
          'assets/app_icon_previews/NATINFoPlus_Pride_Rainbow_512.png',
    ),
    AppIconOption(
      key: 'pride_trans',
      label: 'Pride trans',
      previewAssetPath:
          'assets/app_icon_previews/NATINFoPlus_Pride_Trans_512.png',
    ),
  ];

  static final Set<String> _optionKeys =
      options.map((AppIconOption option) => option.key).toSet();

  static bool _isValidOptionKey(String key) => _optionKeys.contains(key);

  static bool get _isPlatformEligible => Platform.isAndroid || Platform.isIOS;

  /// Returns the option matching [key] or the default option if none matches.
  static AppIconOption getOptionByKey(String key) {
    if (_isValidOptionKey(key)) {
      return options.firstWhere((option) => option.key == key);
    }
    return options.firstWhere((option) => option.key == defaultKey);
  }

  /// Indicates whether the current platform can customise the app icon.
  Future<bool> isSupported() async {
    if (!_isPlatformEligible) {
      return false;
    }
    try {
      final supported = await _channel.invokeMethod<bool>('isSupported');
      return supported ?? false;
    } on PlatformException {
      return false;
    }
  }

  /// Returns the icon key applied on device, synchronising preferences when needed.
  Future<String> getCurrentIconKey() async {
    final prefs = await SharedPreferences.getInstance();
    final savedKey = prefs.getString(_preferenceKey);
    if (savedKey != null && _isValidOptionKey(savedKey)) {
      return savedKey;
    }

    if (!_isPlatformEligible) {
      return defaultKey;
    }

    try {
      final currentKey = await _channel.invokeMethod<String>('getCurrentIcon');
      if (currentKey != null && _isValidOptionKey(currentKey)) {
        await prefs.setString(_preferenceKey, currentKey);
        return currentKey;
      }
    } on PlatformException {
      // Ignore: fallback to default
    }
    return defaultKey;
  }

  /// Requests the platform layer to display the icon identified by [key].
  Future<void> setIcon(String key) async {
    if (!_isValidOptionKey(key)) {
      throw ArgumentError('Icône inconnue: $key');
    }

    if (!_isPlatformEligible) {
      throw UnsupportedError(
        'La personnalisation de l’icône n’est pas disponible sur cette plateforme.',
      );
    }

    await _channel.invokeMethod<void>('setIcon', {'iconKey': key});
    final prefs = await SharedPreferences.getInstance();
    await prefs.setString(_preferenceKey, key);
  }
}
