import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/favourite_folder_service.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';

/// Displays the content of a favourite folder directly inside the favourites
/// tab.
class FavouriteFolderContentView extends StatelessWidget {
  const FavouriteFolderContentView({
    super.key,
    required this.node,
    required this.subtitle,
    required this.onBack,
    required this.onRenameCurrent,
    required this.onDeleteCurrent,
    required this.onRemoveNatinf,
    required this.subfolderCards,
  });

  /// Current folder node being displayed.
  final FavouriteFolderNode node;

  /// Subtitle describing the folder content.
  final String subtitle;

  /// Triggered when the user navigates back to the parent folder.
  final VoidCallback onBack;

  /// Callback used to rename the current folder.
  final Future<void> Function() onRenameCurrent;

  /// Callback used to delete the current folder.
  final Future<void> Function() onDeleteCurrent;

  /// Callback used to remove a NATINF from the current folder.
  final Future<void> Function(Natinf natinf) onRemoveNatinf;

  /// Cards representing the direct children of the current folder.
  final List<Widget> subfolderCards;

  @override
  Widget build(BuildContext context) {
    final children = <Widget>[_buildHeader(context)];

    if (subfolderCards.isNotEmpty) {
      children
        ..addAll(subfolderCards)
        ..add(const SizedBox(height: 8));
    }

    if (node.natinfs.isNotEmpty) {
      children.add(
        Padding(
          padding: const EdgeInsets.symmetric(horizontal: 4),
          child: NatinfList(
            natinfList: node.natinfs,
            nested: true,
            trailingBuilder:
                (context, natinf) => IconButton(
                  icon: const Icon(Icons.remove_circle_outline),
                  tooltip: 'Retirer du dossier',
                  onPressed: () => onRemoveNatinf(natinf),
                ),
          ),
        ),
      );
    }

    if (subfolderCards.isEmpty && node.natinfs.isEmpty) {
      children.add(_buildEmptyState(context));
    }

    return ListView(
      padding: const EdgeInsets.only(bottom: 80),
      children: children,
    );
  }

  Widget _buildHeader(BuildContext context) {
    return Card(
      margin: const EdgeInsets.fromLTRB(12, 12, 12, 8),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          ListTile(
            leading: IconButton(
              icon: const Icon(Icons.arrow_back),
              tooltip: 'Revenir au dossier parent',
              onPressed: onBack,
            ),
            title: Text(node.folder.name),
            subtitle: Text(subtitle),
            trailing: PopupMenuButton<String>(
              onSelected: (value) async {
                switch (value) {
                  case 'rename':
                    await onRenameCurrent();
                    break;
                  case 'delete':
                    await onDeleteCurrent();
                    break;
                }
              },
              itemBuilder:
                  (context) => const [
                    PopupMenuItem(value: 'rename', child: Text('Renommer')),
                    PopupMenuItem(value: 'delete', child: Text('Supprimer')),
                  ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildEmptyState(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 32),
      child: Column(
        mainAxisSize: MainAxisSize.min,
        children: [
          const Icon(Icons.inbox_outlined, size: 48),
          const SizedBox(height: 16),
          const Text('Ce dossier est vide.', textAlign: TextAlign.center),
          const SizedBox(height: 8),
          const Text(
            "Créez un sous-dossier avec le bouton en haut, ou effectuez un appui long sur l'étoile des favoris d'une fiche NATINF pour l'ajouter ici.",
            textAlign: TextAlign.center,
          ),
        ],
      ),
    );
  }
}
