/// Représente un favori avec métadonnées de synchronisation.
class FavouriteItem {
  const FavouriteItem({
    required this.id,
    required this.numero,
    this.folderUuid,
    required this.position,
    required this.updatedAt,
    this.deleted = false,
  });

  /// Identifiant unique (UUID) du favori.
  final String id;

  /// Référence NATINF associée.
  final String numero;

  /// Identifiant du dossier parent, `null` si à la racine.
  final String? folderUuid;

  /// Position du favori dans son dossier.
  final int position;

  /// Date de dernière mise à jour.
  final DateTime updatedAt;

  /// Indique si l'entrée est marquée supprimée.
  final bool deleted;

  /// Crée un favori à partir d'une carte persistée.
  factory FavouriteItem.fromMap(Map<String, dynamic> map) {
    return FavouriteItem(
      id: map['uuid'] as String? ?? map['id'] as String? ?? '',
      numero: map['numero_natinf'] as String? ?? map['numero'] as String? ?? '',
      folderUuid: map['folder_uuid'] as String?,
      position: map['position'] is int ? map['position'] as int : 0,
      updatedAt:
          _parseDate(map['updated_at']) ??
          _parseDate(map['updatedAt']) ??
          DateTime.fromMillisecondsSinceEpoch(
            map['added_at'] is int ? map['added_at'] as int : 0,
          ),
      deleted: map['deleted'] == true,
    );
  }

  /// Convertit le favori en carte pour la persistence.
  Map<String, dynamic> toMap() {
    return {
      'id': id,
      'uuid': id,
      'numero': numero,
      'numero_natinf': numero,
      'folder_uuid': folderUuid,
      'position': position,
      'updatedAt': updatedAt.toIso8601String(),
      'updated_at': updatedAt.millisecondsSinceEpoch,
      'deleted': deleted,
    };
  }

  /// Renvoie une copie modifiée du favori.
  FavouriteItem copyWith({
    String? id,
    String? numero,
    String? folderUuid,
    int? position,
    DateTime? updatedAt,
    bool? deleted,
  }) {
    return FavouriteItem(
      id: id ?? this.id,
      numero: numero ?? this.numero,
      folderUuid: folderUuid ?? this.folderUuid,
      position: position ?? this.position,
      updatedAt: updatedAt ?? this.updatedAt,
      deleted: deleted ?? this.deleted,
    );
  }
}

DateTime? _parseDate(dynamic value) {
  if (value is int) {
    return DateTime.fromMillisecondsSinceEpoch(value);
  }
  if (value is String && value.isNotEmpty) {
    return DateTime.tryParse(value);
  }
  return null;
}
