import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_mappers.dart';

class NatinfCacheService {
  NatinfCacheService({DatabaseHelper? dbHelper})
    : _dbHelper = dbHelper ?? DatabaseHelper();

  final DatabaseHelper _dbHelper;
  final Map<String, bool> _docsProUnauthorized = {};
  final Map<String, bool> _docsProAvailable = {};
  final Map<String, List<NatinfReference>> _savedReferences = {};
  final Map<String, List<NatinfDocPro>> _docsProByNumero = {};

  Future<void> init() async {
    await _dbHelper.initDb();
    _docsProUnauthorized.addAll(await _dbHelper.loadDocsProUnauthorizedMap());
    _docsProAvailable.addAll(await _dbHelper.loadDocsProAvailableMap());
    _savedReferences.addAll(await _dbHelper.loadReferencesMap());
  }

  Future<bool> hasLocalData() async {
    return !await _dbHelper.isEmpty();
  }

  bool isDocsProUnauthorized(String numero) {
    return _docsProUnauthorized[numero] ?? false;
  }

  bool isDocsProAvailable(String numero) {
    return _docsProAvailable[numero] ?? false;
  }

  List<NatinfDocPro>? getDocsProDocs(String numero) {
    return _docsProByNumero[numero];
  }

  List<NatinfReference>? getSavedReferences(String numero) {
    return _savedReferences[numero];
  }

  Future<void> updateDocsProInfo(
    String numero, {
    required bool unauthorized,
    required bool available,
  }) async {
    _docsProUnauthorized[numero] = unauthorized;
    _docsProAvailable[numero] = available;
    await _dbHelper.setDocsProInfo(
      numero,
      unauthorized: unauthorized,
      available: available,
    );
  }

  Future<void> saveReferences(String numero, List<NatinfReference> refs) async {
    _savedReferences[numero] = refs;
    await _dbHelper.setReferences(numero, refs);
  }

  Future<List<CategoryBrief>> getCachedRootCategories() {
    return _dbHelper.getRootCategories();
  }

  Future<Category?> getCachedCategory(int id) async {
    final cat = await _dbHelper.getCategory(id);
    if (cat == null) return null;
    final natinfs = materializeNatinfs(cat.natinfs);
    return cat.copyWith(natinfs: natinfs);
  }

  Future<Natinf?> getNatinfByNumero(String numero) {
    return _dbHelper.getNatinfByNumero(numero);
  }

  Future<int> saveCategory(Category category) {
    return _dbHelper.insertOrUpdateCategory(category);
  }

  Future<void> saveCategories(Iterable<Category> categories) async {
    for (final category in categories) {
      await saveCategory(category);
    }
  }

  Future<int> saveNatinf(Natinf natinf) {
    return _dbHelper.insertOrUpdateNatinf(natinf);
  }

  Future<void> recordHistoryEntry(String numero) {
    return _dbHelper.addHistory(numero);
  }

  Future<List<Natinf>> searchLocal(String query) {
    return _dbHelper.getNatinfByQuery(query);
  }

  Future<void> clearDocsProUnauthorized() async {
    _docsProUnauthorized.clear();
    _docsProAvailable.clear();
    _docsProByNumero.clear();
    await _dbHelper.clearDocsPro();
  }

  Future<void> saveDocsProDocs(String numero, List<NatinfDocPro> docs) async {
    _docsProByNumero[numero] = docs;
  }
}
