import 'dart:convert';

import 'package:http/http.dart' as http;
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:uuid/uuid.dart';

class FavoritesSyncException implements Exception {
  /// Exception levée lors d'un échec de synchronisation des favoris.
  FavoritesSyncException(this.message, {this.statusCode, this.body});

  final String message;
  final int? statusCode;
  final String? body;

  @override
  String toString() => 'FavoritesSyncException($message, status=$statusCode)';
}

class FavoritesSyncService {
  /// Service de synchronisation des favoris avec l'API.
  FavoritesSyncService({
    required Uri baseUri,
    required AuthRepository authRepository,
    DatabaseHelper? dbHelper,
    http.Client? client,
  }) : _baseUri = baseUri,
       _authRepository = authRepository,
       _db = dbHelper ?? DatabaseHelper(),
       _client = client ?? authRepository.authorizedClient;

  final Uri _baseUri;
  final AuthRepository _authRepository;
  final DatabaseHelper _db;
  final http.Client _client;
  final _uuid = const Uuid();

  /// Applique un snapshot serveur dans la base locale (remplacement).
  Future<void> applyServerSnapshot(Map<String, dynamic> snapshot) async {
    final folders = snapshot['folders'] as List<dynamic>? ?? const [];
    final items = snapshot['items'] as List<dynamic>? ?? const [];

    final nowMs = DateTime.now().millisecondsSinceEpoch;
    final folderPayloads = <Map<String, dynamic>>[];
    for (final folder in folders.whereType<Map<String, dynamic>>()) {
      folderPayloads.add({
        'uuid': _coerceString(folder['id']) ?? _uuid.v4(),
        'parent_uuid': _coerceString(folder['parentId']),
        'name': folder['name'] ?? 'Dossier',
        'position': folder['position'] is int ? folder['position'] : 0,
        'updated_at': _toMillis(folder['updatedAt'], fallback: nowMs),
        'created_at': _toMillis(folder['updatedAt'], fallback: nowMs),
        'deleted': folder['deleted'] == true,
      });
    }

    final rootItems = <Map<String, dynamic>>[];
    final folderItemsPayload = <Map<String, dynamic>>[];
    for (final raw in items.whereType<Map<String, dynamic>>()) {
      final numero = raw['numero'] as String? ?? '';
      if (numero.isEmpty) continue;
      final folderUuid = _coerceString(raw['folderId']);
      final payload = {
        'uuid': _coerceString(raw['id']) ?? _uuid.v4(),
        'folder_uuid': folderUuid,
        'numero': numero,
        'position': raw['position'] is int ? raw['position'] : 0,
        'updated_at': _toMillis(raw['updatedAt'], fallback: nowMs),
        'created_at': _toMillis(raw['updatedAt'], fallback: nowMs),
        'deleted': raw['deleted'] == true,
      };
      if (folderUuid == null) {
        rootItems.add(payload);
      } else {
        folderItemsPayload.add(payload);
      }
    }

    await _db.replaceFavoritesData(
      folders: folderPayloads,
      rootItems: rootItems,
      folderItems: folderItemsPayload,
    );
    await _saveStateFromSnapshot(snapshot);
  }

  /// Télécharge l'instantané des favoris depuis le serveur et l'enregistre.
  Future<Map<String, dynamic>> pullFromServer() async {
    await _authRepository.ensureValidAccessToken();
    final uri = _baseUri.resolve('/api/favorites/');
    final response = await _client.get(uri, headers: _jsonHeaders());
    if (response.statusCode != 200) {
      throw FavoritesSyncException(
        'GET favorites failed',
        statusCode: response.statusCode,
        body: response.body,
      );
    }
    final decoded = jsonDecode(response.body) as Map<String, dynamic>;
    final snapshot = decoded['snapshot'] as Map<String, dynamic>? ?? {};
    await applyServerSnapshot(snapshot);
    return decoded;
  }

  /// Applique localement un snapshot déjà téléchargé (sans appel réseau).
  Future<void> applySnapshotLocally(Map<String, dynamic> snapshot) {
    return applyServerSnapshot(snapshot);
  }

  /// Pousse l'état local des favoris vers le serveur.
  Future<Map<String, dynamic>> pushLocal({bool forceReplace = false}) async {
    await _authRepository.ensureValidAccessToken();
    final snapshot = await _buildLocalSnapshot();
    final uri = _baseUri.resolve('/api/favorites/');
    final response = await _client.post(
      uri,
      headers: _jsonHeaders(),
      body: jsonEncode({'snapshot': snapshot, 'forceReplace': forceReplace}),
    );
    if (response.statusCode != 200) {
      throw FavoritesSyncException(
        'POST favorites failed',
        statusCode: response.statusCode,
        body: response.body,
      );
    }
    final decoded = jsonDecode(response.body) as Map<String, dynamic>;
    final serverSnapshot = decoded['snapshot'] as Map<String, dynamic>? ?? {};
    await applyServerSnapshot(serverSnapshot);
    return decoded;
  }

  Future<Map<String, dynamic>> _buildLocalSnapshot() async {
    final folders = await _db.getFavouriteFoldersRaw(includeDeleted: true);
    final rootFavs = await _db.getFavouriteRootRaw(includeDeleted: true);
    final folderItems = await _db.getFavouriteFolderItemsRaw(
      includeDeleted: true,
    );
    final folderUuidById = <int, String>{};

    for (final folder in folders) {
      final id = folder['id'] as int?;
      final uuid = _coerceString(folder['uuid']) ?? _uuid.v4();
      if (id != null) {
        folderUuidById[id] = uuid;
      }
    }

    final folderPayloads =
        folders.map((folder) {
          final id = _coerceString(folder['uuid']) ?? _uuid.v4();
          final parentId = folder['parent_id'] as int?;
          final parentUuid = parentId != null ? folderUuidById[parentId] : null;
          return {
            'id': id,
            'parentId': parentUuid,
            'name': folder['name'] ?? '',
            'position': folder['position'] is int ? folder['position'] : 0,
            'updatedAt': _toIso(folder['updated_at']),
            'deleted': folder['deleted'] == true,
          };
        }).toList();

    final itemPayloads = <Map<String, dynamic>>[];
    for (final fav in rootFavs) {
      itemPayloads.add({
        'id': _coerceString(fav['uuid']) ?? _uuid.v4(),
        'folderId': null,
        'numero': fav['numero_natinf'] ?? '',
        'position': fav['position'] is int ? fav['position'] : 0,
        'updatedAt': _toIso(fav['updated_at'] ?? fav['added_at']),
        'deleted': fav['deleted'] == true,
      });
    }
    for (final item in folderItems) {
      final folderId = item['folder_id'] as int?;
      final folderUuid =
          item['folder_uuid'] as String? ??
          (folderId != null ? folderUuidById[folderId] : null);
      itemPayloads.add({
        'id': _coerceString(item['uuid']) ?? _uuid.v4(),
        'folderId': folderUuid,
        'numero': item['numero_natinf'] ?? '',
        'position': item['position'] is int ? item['position'] : 0,
        'updatedAt': _toIso(item['updated_at'] ?? item['added_at']),
        'deleted': item['deleted'] == true,
      });
    }

    final state = await _db.getFavoritesState();
    final version = state['version'] as String?;
    return {
      'version': version ?? _uuid.v4(),
      'generated_at': DateTime.now().toIso8601String(),
      'folders': folderPayloads,
      'items': itemPayloads,
    };
  }

  Future<void> _saveStateFromSnapshot(Map<String, dynamic> snapshot) async {
    final state = await _db.getFavoritesState();
    final newState = Map<String, dynamic>.from(state);
    newState['version'] = snapshot['version'];
    newState['checksum'] = snapshot['checksum'];
    newState['lastSyncAt'] = DateTime.now().toIso8601String();
    newState['deviceId'] = await _deviceId();
    await _db.saveFavoritesState(newState);
  }

  Future<String> _deviceId() async {
    final state = await _db.getFavoritesState();
    final existing = _coerceString(state['deviceId']);
    if (existing != null && existing.isNotEmpty) {
      return existing;
    }
    final generated = _uuid.v4();
    final newState = Map<String, dynamic>.from(state);
    newState['deviceId'] = generated;
    await _db.saveFavoritesState(newState);
    return generated;
  }

  Map<String, String> _jsonHeaders() {
    return {'Content-Type': 'application/json', 'Accept': 'application/json'};
  }
}

String _toIso(dynamic value) {
  if (value is int) {
    return DateTime.fromMillisecondsSinceEpoch(value).toIso8601String();
  }
  if (value is String && value.isNotEmpty) {
    return DateTime.tryParse(value)?.toIso8601String() ?? value;
  }
  return DateTime.now().toIso8601String();
}

String? _coerceString(dynamic value) {
  if (value == null) return null;
  if (value is String) return value;
  return value.toString();
}

int _toMillis(dynamic value, {required int fallback}) {
  if (value is int) {
    return value;
  }
  if (value is String && value.isNotEmpty) {
    final parsed = DateTime.tryParse(value);
    if (parsed != null) {
      return parsed.millisecondsSinceEpoch;
    }
  }
  return fallback;
}
