import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';

/// Handles theme and infraction color preferences.
class ThemeProvider extends ChangeNotifier {
  static const Color _defaultPrimaryColor = Colors.indigo;
  static const Color _defaultObligatoryColor = Color.fromRGBO(255, 0, 0, 0.5);
  static const Color _defaultFacultativeColor = Color.fromRGBO(
    255,
    193,
    7,
    0.5,
  );
  static const Color _defaultNonColor = Color.fromRGBO(158, 158, 158, 0.5);
  static const Color _defaultCrimeColor = Color.fromRGBO(255, 0, 0, 0.5);
  static const Color _defaultDelitColor = Color.fromRGBO(255, 193, 7, 0.5);
  static const Color _defaultContraventionColor = Color.fromRGBO(
    76,
    175,
    80,
    0.5,
  );

  bool _isDarkMode = false;

  /// Whether the dark theme is active.
  bool get isDarkMode => _isDarkMode;

  Color _primaryColor = _defaultPrimaryColor;

  /// Current accent color used by the app.
  Color get primaryColor => _primaryColor;

  /// Text color that contrasts with [primaryColor].
  Color get primaryTextColor =>
      _primaryColor.computeLuminance() < 0.5 ? Colors.white : Colors.black;

  Color _obligatoryColor = _defaultObligatoryColor;

  /// Badge color for "obligatoire" infractions.
  Color get obligatoryColor => _obligatoryColor;

  Color _facultativeColor = _defaultFacultativeColor;

  /// Badge color for "facultatif" infractions.
  Color get facultativeColor => _facultativeColor;

  Color _nonColor = _defaultNonColor;

  /// Badge color for "non" infractions.
  Color get nonColor => _nonColor;

  Color _crimeColor = _defaultCrimeColor;

  /// Badge color for crimes.
  Color get crimeColor => _crimeColor;

  Color _delitColor = _defaultDelitColor;

  /// Badge color for délits.
  Color get delitColor => _delitColor;

  Color _contraventionColor = _defaultContraventionColor;

  /// Badge color for contraventions.
  Color get contraventionColor => _contraventionColor;

  /// Loads theme preferences from shared storage.
  Future<void> load({SharedPreferences? prefs, bool notify = true}) async {
    final sharedPrefs = prefs ?? await SharedPreferences.getInstance();
    _applyPrefs(sharedPrefs);
    if (notify) {
      notifyListeners();
    }
  }

  void _applyPrefs(SharedPreferences prefs) {
    _isDarkMode = prefs.getBool('isDarkMode') ?? false;
    _primaryColor = Color(
      prefs.getInt('primaryColor') ??
          (_isDarkMode
              ? Colors.black.toARGB32()
              : _defaultPrimaryColor.toARGB32()),
    );
    _obligatoryColor = Color(
      prefs.getInt('obligatoryColor') ?? _defaultObligatoryColor.toARGB32(),
    );
    _facultativeColor = Color(
      prefs.getInt('facultativeColor') ?? _defaultFacultativeColor.toARGB32(),
    );
    _nonColor = Color(prefs.getInt('nonColor') ?? _defaultNonColor.toARGB32());
    _crimeColor = Color(
      prefs.getInt('crimeColor') ?? _defaultCrimeColor.toARGB32(),
    );
    _delitColor = Color(
      prefs.getInt('delitColor') ?? _defaultDelitColor.toARGB32(),
    );
    _contraventionColor = Color(
      prefs.getInt('contraventionColor') ??
          _defaultContraventionColor.toARGB32(),
    );
  }

  /// Sets the dark theme flag and persists it.
  Future<void> toggleTheme(bool isDark) async {
    _isDarkMode = isDark;
    final prefs = await SharedPreferences.getInstance();
    await prefs.setBool('isDarkMode', _isDarkMode);
    _applyPrefs(prefs);
    notifyListeners();
  }

  /// Returns a badge color associated with the provided infraction label.
  Color colorForInfractionLabel(String label) {
    final lower = label.trim().toLowerCase();
    if (lower == 'obligatoire') return _obligatoryColor;
    if (lower == 'facultatif') return _facultativeColor;
    if (lower == 'non') return _nonColor;
    if (lower.contains('crime')) return _crimeColor;
    if (lower.contains('delit') || lower.contains('délit')) return _delitColor;
    if (lower.contains('contravention')) return _contraventionColor;
    return Colors.transparent;
  }

  /// Whether the label has a known badge color.
  bool hasInfractionBadge(String label) {
    final lower = label.trim().toLowerCase();
    return lower == 'obligatoire' ||
        lower == 'facultatif' ||
        lower == 'non' ||
        lower.contains('crime') ||
        lower.contains('delit') ||
        lower.contains('délit') ||
        lower.contains('contravention');
  }

  Future<void> _setColor(
    String key,
    Color value,
    void Function(Color color) assign,
  ) async {
    assign(value);
    final prefs = await SharedPreferences.getInstance();
    await prefs.setInt(key, value.toARGB32());
    notifyListeners();
  }

  /// Updates the primary accent color.
  Future<void> updatePrimaryColor(Color newColor) async {
    await _setColor('primaryColor', newColor, (color) => _primaryColor = color);
  }

  /// Updates the obligatory badge color.
  Future<void> updateObligatoryColor(Color newColor) async {
    await _setColor(
      'obligatoryColor',
      newColor,
      (color) => _obligatoryColor = color,
    );
  }

  /// Updates the facultative badge color.
  Future<void> updateFacultativeColor(Color newColor) async {
    await _setColor(
      'facultativeColor',
      newColor,
      (color) => _facultativeColor = color,
    );
  }

  /// Updates the "non" badge color.
  Future<void> updateNonColor(Color newColor) async {
    await _setColor('nonColor', newColor, (color) => _nonColor = color);
  }

  /// Updates the crime badge color.
  Future<void> updateCrimeColor(Color newColor) async {
    await _setColor('crimeColor', newColor, (color) => _crimeColor = color);
  }

  /// Updates the délit badge color.
  Future<void> updateDelitColor(Color newColor) async {
    await _setColor('delitColor', newColor, (color) => _delitColor = color);
  }

  /// Updates the contravention badge color.
  Future<void> updateContraventionColor(Color newColor) async {
    await _setColor(
      'contraventionColor',
      newColor,
      (color) => _contraventionColor = color,
    );
  }

  /// Resets all infraction badge colors to their defaults.
  Future<void> resetInfractionColors() async {
    await Future.wait([
      updateObligatoryColor(_defaultObligatoryColor),
      updateFacultativeColor(_defaultFacultativeColor),
      updateNonColor(_defaultNonColor),
      updateCrimeColor(_defaultCrimeColor),
      updateDelitColor(_defaultDelitColor),
      updateContraventionColor(_defaultContraventionColor),
    ]);
  }
}
