import 'dart:typed_data';

import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/shared/data_sources/asset_source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

void main() {
  group('AssetSourceAdapter', () {
    test('loads bytes from asset bundle', () async {
      final adapter = AssetSourceAdapter(
        bundle: _FakeAssetBundle({'data.json': 'asset-data'.codeUnits}),
      );
      final spec = SourceSpec(
        id: 'asset',
        dataset: 'natinf',
        name: 'Infractions NATINFo',
        type: SourceType.asset,
        uri: Uri.parse('asset://data.json'),
        scope: 'natinf',
        priority: 1,
        requiredAtBuild: false,
        requiresNetwork: false,
        schemaVersion: 1,
        checksum: '428287e7064a6dc838860c0dcdac5d03',
        checksumAlgo: 'md5',
      );

      final result = await adapter.load(spec);
      expect(result.bytes, 'asset-data'.codeUnits);
      expect(result.integrity.status, ChecksumStatus.match);
    });

    test('marks mismatch when checksum differs', () async {
      final adapter = AssetSourceAdapter(
        bundle: _FakeAssetBundle({'data.json': 'asset-data'.codeUnits}),
      );
      final spec = SourceSpec(
        id: 'asset',
        dataset: 'natinf',
        name: 'Infractions NATINFo',
        type: SourceType.asset,
        uri: Uri.parse('asset://data.json'),
        scope: 'natinf',
        priority: 1,
        requiredAtBuild: false,
        requiresNetwork: false,
        schemaVersion: 1,
        checksum: 'deadbeef',
        checksumAlgo: 'md5',
      );

      final result = await adapter.load(spec);
      expect(result.integrity.status, ChecksumStatus.mismatch);
    });
  });
}

class _FakeAssetBundle extends CachingAssetBundle {
  _FakeAssetBundle(this.assets);

  final Map<String, List<int>> assets;

  @override
  Future<ByteData> load(String key) async {
    final bytes = assets[key];
    if (bytes == null) {
      throw FlutterError('Asset not found: $key');
    }
    return ByteData.view(Uint8List.fromList(bytes).buffer);
  }
}
