import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_mappers.dart';

void main() {
  group('materializeNatinfs', () {
    test('returns empty list for null input', () {
      final result = materializeNatinfs(null);
      expect(result, isEmpty);
    });

    test('preserves existing Natinf instances', () {
      const original = Natinf(id: 1, numeroNatinf: '100', dacgSource: 0);
      final result = materializeNatinfs(const [original]);
      expect(result.single, same(original));
    });

    test('converts raw map entries into Natinf', () {
      final raw = [
        const Natinf(id: 1, numeroNatinf: '100', dacgSource: 0),
        {
          'id': 2,
          'numero_natinf': '200',
          'qualification_infraction': 'Test',
          'dacg_source': 0,
        },
      ];

      final result = materializeNatinfs(raw);

      expect(result.length, 2);
      expect(result.first.numeroNatinf, '100');
      expect(result.last.numeroNatinf, '200');
      expect(result.last.qualificationInfraction, 'Test');
    });
  });

  group('parseStoredNatinf', () {
    test('adds default dacg_source when missing', () {
      final natinf = parseStoredNatinf({
        'id': 3,
        'numero_natinf': '300',
        'qualification_infraction': 'Legacy entry',
      });

      expect(natinf.numeroNatinf, '300');
      expect(natinf.dacgSource, 0);
      expect(natinf.qualificationInfraction, 'Legacy entry');
    });
  });

  group('parseServerNatinf', () {
    test('strips transient fields and returns ParsedServerNatinf', () {
      final parsed = parseServerNatinf({
        'id': 42,
        'numero_natinf': '777',
        'docs_pro_available': 'true',
        'docs_pro': {'error': 'Unauthorized'},
        'references': [
          {'title': 'L1', 'url': 'https://example.test/ref'},
        ],
        'categories': <Map<String, dynamic>>[],
        'dacg_source': 1,
      });

      expect(parsed.natinf.id, 42);
      expect(parsed.numero, '777');
      expect(parsed.docsProAvailable, isTrue);
      expect(parsed.docsProUnauthorized, isTrue);
      expect(parsed.references.single.title, 'L1');
      expect(parsed.references.single.url, 'https://example.test/ref');
    });

    test('handles missing optional fields gracefully', () {
      final parsed = parseServerNatinf({
        'id': 1,
        'numero_natinf': '100',
        'qualification_infraction': 'Test',
        'dacg_source': 0,
      });

      expect(parsed.docsProAvailable, isFalse);
      expect(parsed.docsProUnauthorized, isFalse);
      expect(parsed.references, isEmpty);
      expect(parsed.natinf.qualificationInfraction, 'Test');
    });
  });
}
