import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:sembast/sembast_memory.dart';

void main() {
  late DatabaseHelper databaseHelper;

  setUp(() async {
    databaseHelper = DatabaseHelper();
    final dbPath = 'memory_${DateTime.now().microsecondsSinceEpoch}.db';
    await databaseHelper.initDbForTesting(databaseFactoryMemory, dbPath);
  });

  test('addFavourites persists sync metadata and filters tombstones', () async {
    await databaseHelper.clearFavourites();
    await databaseHelper.addFavourites('42');

    final store = intMapStoreFactory.store('favourites');
    await store.add(databaseHelper.dbForTesting, {
      'numero_natinf': '99',
      'added_at': 0,
      'deleted': true,
    });

    final favourites = await databaseHelper.getFavourites();
    expect(favourites, hasLength(1));

    final favourite = favourites.first;
    expect(favourite['uuid'], isNotEmpty);
    expect(favourite['id'], favourite['uuid']);
    expect(favourite['position'], 0);
    expect(favourite['deleted'], isFalse);
    expect(favourite['updated_at'], isA<int>());
  });

  test(
    'createFavouriteFolder stores metadata and propagates folder uuid to items',
    () async {
      final folderId = await databaseHelper.createFavouriteFolder(
        name: 'Racine',
      );
      final folders = await databaseHelper.getAllFavouriteFolders();
      final folder = folders.firstWhere(
        (candidate) => candidate.id == folderId,
      );

      expect(folder.uuid, isNotNull);
      expect(folder.updatedAt, isNotNull);
      expect(folder.position, 0);

      await databaseHelper.addNatinfToFolder(folderId, '900');
      final itemsStore = stringMapStoreFactory.store('favourite_folder_items');
      final records = await itemsStore.find(databaseHelper.dbForTesting);

      expect(records, hasLength(1));

      final map = records.first.value as Map<String, dynamic>;
      expect(map['folder_uuid'], folder.uuid);
      expect(map['uuid'], isNotEmpty);
      expect(map['position'], 0);
      expect(map['deleted'], isFalse);
      expect(map['updated_at'], isA<int>());
    },
  );
}
