import 'package:shared_preferences/shared_preferences.dart';

/// Persists and reads user-defined overrides for data sources.
///
/// Keys follow the `customSourceUrls_<dataset>` convention.
class SourceOverrideStore {
  SourceOverrideStore(this._prefs);

  final SharedPreferences _prefs;

  static const String keyPrefix = 'customSourceUrls_';

  /// Returns override URLs keyed by dataset identifier.
  Map<String, Uri> loadOverrides() {
    final overrides = <String, Uri>{};
    for (final key in _prefs.getKeys()) {
      if (!key.startsWith(keyPrefix)) continue;
      final value = _prefs.getString(key);
      if (value == null) continue;
      final trimmed = value.trim();
      if (trimmed.isEmpty) continue;
      final uri = Uri.tryParse(trimmed);
      if (uri == null) continue;
      final dataset = key.substring(keyPrefix.length);
      if (dataset.isEmpty) continue;
      overrides[dataset] = uri;
    }
    return overrides;
  }

  /// Persists [uri] as the override for [dataset].
  Future<void> saveOverride(String dataset, Uri uri) {
    return _prefs.setString(keyFor(dataset), uri.toString());
  }

  /// Returns the preference key used to store the override for [dataset].
  static String keyFor(String dataset) => '$keyPrefix$dataset';
}
