import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_map/flutter_map.dart';
import 'package:flutter_map_compass/flutter_map_compass.dart';
import 'package:latlong2/latlong.dart';
import 'package:provider/provider.dart';
import 'package:url_launcher/url_launcher.dart';

import 'package:natinfo_flutter/features/pr_locator/presentation/viewmodels/pr_locator_viewmodel.dart';

class PrMap extends StatefulWidget {
  final MapController controller;
  const PrMap({required this.controller, super.key});

  @override
  _PrMapState createState() => _PrMapState();
}

class _PrMapState extends State<PrMap> {
  PrLocatorViewModel? _vm;
  bool _mapReady = false;

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    final newVm = context.watch<PrLocatorViewModel>();
    if (_vm != newVm) {
      _vm?.removeListener(_onMarkersChanged);
      _vm = newVm;
      _vm!.addListener(_onMarkersChanged);
    }
  }

  void _onMarkersChanged() {
    if (!_mapReady) return;
    final pts = _vm!.markers;
    if (pts.isEmpty) return;

    // Si un seul point (ou tous identiques), zoom fixe
    bool degenerate = pts.every((pt) => pt == pts.first);
    if (degenerate) {
      WidgetsBinding.instance.addPostFrameCallback((_) {
        widget.controller.move(pts.first, 15);
      });
    } else {
      // Sinon ajuster aux deux
      final bounds = LatLngBounds.fromPoints(pts);
      WidgetsBinding.instance.addPostFrameCallback((_) {
        widget.controller.fitCamera(
          CameraFit.bounds(bounds: bounds, padding: const EdgeInsets.all(50)),
        );
      });
    }
  }

  @override
  void dispose() {
    _vm?.removeListener(_onMarkersChanged);
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final vm = context.watch<PrLocatorViewModel>();

    // Affiche la bottom sheet adaptée
    void showInfoSheet({required LatLng pt, required bool isUser}) {
      showModalBottomSheet<void>(
        context: context,
        isScrollControlled: true,
        shape: const RoundedRectangleBorder(
          borderRadius: BorderRadius.vertical(top: Radius.circular(16)),
        ),
        builder: (_) {
          return Padding(
            padding: EdgeInsets.only(
              bottom: MediaQuery.of(context).viewInsets.bottom,
            ),
            child: isUser ? _buildLocationSheet(pt) : _buildPrSheet(pt, vm),
          );
        },
      );
    }

    // Construction des markers avec GestureDetector
    final markerWidgets =
        vm.markers.map((pt) {
          final bool isUser = pt == vm.markers.first;
          return Marker(
            width: 40,
            height: 40,
            point: pt,
            child: GestureDetector(
              onTap: () => showInfoSheet(pt: pt, isUser: isUser),
              child: Icon(
                isUser ? Icons.my_location : Icons.flag,
                color: isUser ? Colors.blue : Colors.red,
                size: 32,
              ),
            ),
          );
        }).toList();

    return FlutterMap(
      mapController: widget.controller,
      options: MapOptions(
        initialCenter: const LatLng(46.2276, 2.2137),
        initialZoom: 6,
        onMapReady: () => setState(() => _mapReady = true),
        // Tap hors icônes → recalcul nearest PR
        onTap: (_, pt) => vm.handleTap(pt),
      ),
      children: [
        TileLayer(
          urlTemplate: 'https://tile.openstreetmap.org/{z}/{x}/{y}.png',
          subdomains: const ['a', 'b', 'c'],
          userAgentPackageName: 'net.retiolus.natinfo',
        ),
        RichAttributionWidget(
          attributions: [
            // Suggested attribution for the OpenStreetMap public tile server
            TextSourceAttribution(
              'OpenStreetMap contributors',
              onTap:
                  () => launchUrl(
                    Uri.parse('https://openstreetmap.org/copyright'),
                  ),
            ),
          ],
        ),
        const MapCompass.cupertino(hideIfRotatedNorth: true),
        MarkerLayer(markers: markerWidgets),
      ],
    );
  }

  Widget _buildLocationSheet(LatLng pt) {
    final coordText =
        'lat ${pt.latitude.toStringAsFixed(6)}, lon ${pt.longitude.toStringAsFixed(6)}';
    return Container(
      padding: const EdgeInsets.all(16),
      child: Column(
        mainAxisSize: MainAxisSize.min,
        children: [
          Text('Ma position', style: Theme.of(context).textTheme.titleLarge),
          const SizedBox(height: 8),
          Text(coordText),
          const SizedBox(height: 16),
          Row(
            mainAxisAlignment: MainAxisAlignment.spaceEvenly,
            children: [
              ElevatedButton.icon(
                icon: const Icon(Icons.open_in_new),
                label: const Text('Ouvrir'),
                onPressed: () {
                  final uri = Uri.parse('geo:${pt.latitude},${pt.longitude}');
                  launchUrl(uri, mode: LaunchMode.externalApplication);
                },
              ),
              ElevatedButton.icon(
                icon: const Icon(Icons.copy),
                label: const Text('Copier'),
                onPressed: () {
                  Clipboard.setData(ClipboardData(text: coordText));
                  Navigator.pop(context);
                },
              ),
              ElevatedButton.icon(
                icon: const Icon(Icons.close),
                label: const Text('Fermer'),
                onPressed: () => Navigator.pop(context),
              ),
            ],
          ),
        ],
      ),
    );
  }

  Widget _buildPrSheet(LatLng pt, PrLocatorViewModel vm) {
    final list = vm.datasets[vm.selectedDataset]!;
    final pr = list.firstWhere(
      (p) => p.lat == pt.latitude && p.lon == pt.longitude,
      orElse: () => throw StateError('PR point non trouvé'),
    );
    final infoText = vm.result ?? 'PR ${pr.pr} (${pr.route})';
    final coordText =
        'lat ${pr.lat.toStringAsFixed(6)}, lon ${pr.lon.toStringAsFixed(6)}';

    return Container(
      padding: const EdgeInsets.all(16),
      child: Column(
        mainAxisSize: MainAxisSize.min,
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text('Point routier', style: Theme.of(context).textTheme.titleLarge),
          const SizedBox(height: 8),
          Text(infoText),
          const SizedBox(height: 4),
          Text(coordText),
          const SizedBox(height: 16),
          Row(
            mainAxisAlignment: MainAxisAlignment.spaceEvenly,
            children: [
              ElevatedButton.icon(
                icon: const Icon(Icons.open_in_new),
                label: const Text('Ouvrir'),
                onPressed: () {
                  final uri = Uri.parse('geo:${pr.lat},${pr.lon}');
                  launchUrl(uri, mode: LaunchMode.externalApplication);
                },
              ),
              ElevatedButton.icon(
                icon: const Icon(Icons.copy),
                label: const Text('Copier'),
                onPressed: () {
                  Clipboard.setData(
                    ClipboardData(text: '$infoText\n$coordText'),
                  );
                  Navigator.pop(context);
                },
              ),
              ElevatedButton.icon(
                icon: const Icon(Icons.close),
                label: const Text('Fermer'),
                onPressed: () => Navigator.pop(context),
              ),
            ],
          ),
        ],
      ),
    );
  }
}
