import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:url_launcher/url_launcher.dart';

/// Displays a bottom sheet with the amounts for the infraction.
/// If the infraction is a contravention (amende_category AF):
///   - If montant_af_min, montant_af and montant_af_maj are provided,
///     shows those.
///   - Otherwise, shows the fallback general amounts for 1re à 5e classe.
///     If the nature field contains a digit (ex: "1", "2", etc.), only the corresponding fallback is displayed.
/// If the infraction is a délit (amende_category AFD), shows:
///   montant_afd_min, montant_afd, montant_afd_maj.
void showAmendePopup(BuildContext context, Natinf natinf) {
  String title;
  List<Widget> amountTiles = [];
  final nature = natinf.natureInfraction?.toLowerCase() ?? '';

  if (nature.contains('contravention')) {
    // Check if any of the AF amounts are provided.
    final bool hasAfValues =
        (natinf.montantAfMin != null &&
            natinf.montantAfMin!.trim().isNotEmpty) ||
        (natinf.montantAf != null && natinf.montantAf!.trim().isNotEmpty) ||
        (natinf.montantAfMaj != null && natinf.montantAfMaj!.trim().isNotEmpty);

    if (hasAfValues) {
      title = "Montants de l'amende forfaitaire (AF)";
      amountTiles.addAll([
        ListTile(
          title: const Text("Minorée"),
          subtitle: Text(natinf.montantAfMin ?? "N/A"),
        ),
        ListTile(
          title: const Text("Ordinaire"),
          subtitle: Text(natinf.montantAf ?? "N/A"),
        ),
        ListTile(
          title: const Text("Majorée"),
          subtitle: Text(natinf.montantAfMaj ?? "N/A"),
        ),
      ]);
    } else {
      title = "Montants cas général de l'amende forfaitaire (AF)";
      final fallbackAmounts = [
        {"classe": "1re classe", "ordinaire": "11 €", "majoree": "33 €"},
        {"classe": "2e classe", "ordinaire": "35 €", "majoree": "75 €"},
        {"classe": "3e classe", "ordinaire": "68 €", "majoree": "180 €"},
        {"classe": "4e classe", "ordinaire": "135 €", "majoree": "375 €"},
        {"classe": "5e classe", "ordinaire": "200 €", "majoree": "450 €"},
      ];

      final amountsToDisplay =
          fallbackAmounts.where((data) {
            final digit = data["classe"]!.substring(0, 1);
            return nature.contains(digit);
          }).toList();

      // If no specific digit found, display all fallback amounts.
      final displayList =
          amountsToDisplay.isNotEmpty ? amountsToDisplay : fallbackAmounts;

      for (var classeData in displayList) {
        amountTiles.add(
          ListTile(
            title: Text(classeData["classe"]!),
            subtitle: Text(
              "Ordinaire: ${classeData["ordinaire"]}, Majorée: ${classeData["majoree"]}",
            ),
          ),
        );
      }
      amountTiles.add(
        Padding(
          padding: const EdgeInsets.only(top: 8),
          child: InkWell(
            onTap: () async {
              final uri = Uri.parse(
                "https://www.service-public.fr/particuliers/vosdroits/F32803",
              );
              if (await canLaunchUrl(uri)) {
                await launchUrl(uri);
              }
            },
            child: const Text(
              "Ces montants sont ceux cas général. Pour plus d'informations, cliquez ici.",
              style: TextStyle(
                fontSize: 12,
                fontStyle: FontStyle.italic,
                color: Colors.blue,
                decoration: TextDecoration.underline,
              ),
            ),
          ),
        ),
      );
    }
  } else if (nature.contains('délit') &&
      natinf.amendeCategory == NatinfAmendeCategory.afd) {
    title = "Montants de l'amende forfaitaire délictuelle (AFD)";
    amountTiles.addAll([
      ListTile(
        title: const Text("Minorée"),
        subtitle: Text(natinf.montantAfdMin ?? "N/A"),
      ),
      ListTile(
        title: const Text("Ordinaire"),
        subtitle: Text(natinf.montantAfd ?? "N/A"),
      ),
      ListTile(
        title: const Text("Majorée"),
        subtitle: Text(natinf.montantAfdMaj ?? "N/A"),
      ),
    ]);
  } else {
    // If conditions are not met, don't display anything.
    return;
  }

  showModalBottomSheet<void>(
    context: context,
    shape: const RoundedRectangleBorder(
      borderRadius: BorderRadius.vertical(top: Radius.circular(16)),
    ),
    builder: (_) {
      return SafeArea(
        child: SingleChildScrollView(
          child: Padding(
            padding: const EdgeInsets.all(16.0),
            child: Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                Text(title, style: Theme.of(context).textTheme.titleLarge),
                const SizedBox(height: 8),
                ...amountTiles,
                const SizedBox(height: 16),
                ElevatedButton(
                  onPressed: () => Navigator.pop(context),
                  child: const Text("Fermer"),
                ),
              ],
            ),
          ),
        ),
      );
    },
  );
}
