import 'package:diacritic/diacritic.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';

enum NatinfSortKey { apiOrder, natinf, alphabetical, nature }

class NatinfSorter {
  const NatinfSorter(this.rawResults);

  final List<Natinf> rawResults;

  List<Natinf> sort(NatinfSortKey key, {required bool ascending}) {
    final base = List<Natinf>.from(rawResults);
    if (key == NatinfSortKey.apiOrder) {
      return ascending ? base : base.reversed.toList(growable: false);
    }

    final comparator = _comparators[key];
    if (comparator == null) {
      return base;
    }
    base.sort(comparator);
    return ascending ? base : base.reversed.toList(growable: false);
  }

  static final Map<NatinfSortKey, Comparator<Natinf>> _comparators = {
    NatinfSortKey.natinf: _compareNatinf,
    NatinfSortKey.alphabetical: _compareAlphabetical,
    NatinfSortKey.nature: _compareNature,
  };

  static int _compareAlphabetical(Natinf a, Natinf b) {
    final aValue = _normalizedText(
      a.qualificationInfraction ?? a.natureInfraction,
    );
    final bValue = _normalizedText(
      b.qualificationInfraction ?? b.natureInfraction,
    );
    final cmp = aValue.compareTo(bValue);
    if (cmp != 0) {
      return cmp;
    }
    return (a.numeroNatinf ?? '').compareTo(b.numeroNatinf ?? '');
  }

  static int? _natinfNumber(Natinf item) =>
      int.tryParse(item.numeroNatinf ?? '');

  static int _compareNatinf(Natinf a, Natinf b) {
    final aValue = _natinfNumber(a);
    final bValue = _natinfNumber(b);
    if (aValue == null && bValue == null) {
      return _compareAlphabetical(a, b);
    }
    if (aValue == null) {
      return 1;
    }
    if (bValue == null) {
      return -1;
    }
    final cmp = aValue.compareTo(bValue);
    return cmp != 0 ? cmp : _compareAlphabetical(a, b);
  }

  static String _natureValue(Natinf item) =>
      _normalizedText(item.natureInfraction);

  static int _compareNature(Natinf a, Natinf b) {
    final cmp = _natureValue(a).compareTo(_natureValue(b));
    return cmp != 0 ? cmp : _compareAlphabetical(a, b);
  }

  static String _normalizedText(String? value) =>
      removeDiacritics((value ?? '').toLowerCase());
}
