import 'dart:convert';

import 'package:flutter/material.dart';
import 'package:flutter/gestures.dart';
import 'package:flutter/services.dart';
import 'package:flutter_html/flutter_html.dart';
import 'package:http/http.dart' as http;
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/shared/data_sources/source_override_store.dart';

class NatinfLegalTextsPage extends StatefulWidget {
  final Natinf natinf;
  const NatinfLegalTextsPage({super.key, required this.natinf});

  @override
  State<NatinfLegalTextsPage> createState() => _NatinfLegalTextsPageState();
}

class _ArticlePanelItem {
  _ArticlePanelItem(this.title, {this.codeAbbrev});

  final String title;
  final String? codeAbbrev;
  bool isExpanded = false;
  bool isLoading = false;
  String? text;
  String? label;
  String? articleId;
  String? searchQuery;

  String? get url =>
      articleId != null
          ? 'https://www.legifrance.gouv.fr/codes/article_lc/'
              '$articleId'
          : null;
}

class _ArticleInfo {
  const _ArticleInfo({
    required this.text,
    this.label,
    this.id,
    this.searchQuery,
  });

  final String text;
  final String? label;
  final String? id;
  final String? searchQuery;
}

class _NatinfLegalTextsPageState extends State<NatinfLegalTextsPage> {
  late Swagger _api;
  bool _apiReady = false;
  List<_ArticlePanelItem> _definieItems = [];
  List<_ArticlePanelItem> _reprimeeItems = [];
  bool _offline = false;
  bool _loading = true;

  @override
  void initState() {
    super.initState();
    _initApi();
    MatomoService().trackPage(
      title: 'Textes juridiques NATINF ${widget.natinf.numeroNatinf}',
      path: '/natinf/${widget.natinf.numeroNatinf}/textes/',
    );
  }

  Future<void> _initApi() async {
    final prefs = await SharedPreferences.getInstance();
    final apiSource =
        prefs.getString(SourceOverrideStore.keyFor('natinf')) ??
        'https://natinfo.app/api';
    _api = Swagger.create(baseUrl: Uri.parse(apiSource));
    setState(() {
      _apiReady = true;
    });
    await _checkConnectivityAndLoad();
  }

  static const String _offlineMessage =
      'Connexion requise : impossible de consulter les articles (aucune connexion Internet ou mode hors ligne activé).';

  Future<bool> _hasNetwork() async {
    if (AppConfig.forceOffline) return false;
    final result = await Connectivity().checkConnectivity();
    return result != ConnectivityResult.none;
  }

  Future<void> _checkConnectivityAndLoad() async {
    setState(() {
      _loading = true;
    });
    final hasNet = await _hasNetwork();
    if (!hasNet) {
      setState(() {
        _offline = true;
        _loading = false;
      });
      return;
    }
    await _loadArticles();
    setState(() {
      _offline = false;
      _loading = false;
    });
  }

  Future<void> _loadArticles() async {
    try {
      final resp = await _api.natinfoNatinfNumeroNatinfArticlesGet(
        numeroNatinf: widget.natinf.numeroNatinf,
      );
      final Map<String, dynamic> data =
          jsonDecode(resp.bodyString) as Map<String, dynamic>;
      final definie = (data['definie_articles_data'] as List?) ?? [];
      final reprimee = (data['reprimee_articles_data'] as List?) ?? [];
      _definieItems =
          definie
              .map(
                (e) => _ArticlePanelItem(
                  (e['article'] ?? '').toString(),
                  codeAbbrev: e['code_abbrev']?.toString(),
                ),
              )
              .toList();
      _reprimeeItems =
          reprimee
              .map(
                (e) => _ArticlePanelItem(
                  (e['article'] ?? '').toString(),
                  codeAbbrev: e['code_abbrev']?.toString(),
                ),
              )
              .toList();
    } catch (_) {
      _definieItems = [];
      _reprimeeItems = [];
    }
  }

  static const List<String> _abbreviations = [
    'C.ARTISANAT',
    'C.A.S.F',
    'C.ASSURANCES',
    'C.AVIATION',
    'C.CINEMA',
    'C.CIVIL',
    'C.COMMERCE',
    'C.CONSOMMAT',
    'C.CONSTRUCT',
    'C.DEFENSE',
    'C.DOUANES',
    'C.EDUCATION',
    'C.ELECTORAL',
    'C.ENERGIE',
    'C.ENVIR',
    'C.E.S.E.D.A',
    'C.FORESTIER',
    'C.G.C.T',
    'C.G.F.P',
    'C.G.I',
    'C.G.P.P.P',
    'C.I.B.S',
    'C.J.A',
    'C.J.F',
    'C.J.M',
    'C.J.P.M',
    'C.LEGION.H',
    'C.M.F',
    'C.MINIER',
    'C.MUTUALITE',
    'C.PATRIMOINE',
    'C.P.C',
    'C.P&CE',
    'C.P.C.EX',
    'C.PENAL',
    'C.PENS.CIV',
    'C.P.LOCAL',
    'C.P.P',
    'C.PROPR.INT',
    'C.RECHERCHE',
    'C.ROUTE',
    'C.RURAL',
    'C.SANTE.PUB',
    'C.SECU.SOC',
    'C.SERV.NAT',
    'C.S.I',
    'C.SPORT',
    'C.TOURISME',
    'C.TRANSPORTS',
    'C.TRAVAIL',
    'C.URBANISME',
    'C.VOIRIE.R',
    'L.PROC.FISC',
    'ARR.MINIST',
    'ARR.PREFEC',
    'DECRET',
    'DECRET-LOI',
    'LOI',
    'LOI-CONST',
    'LOI-ORG',
    'ORD',
    'ACCORD',
    'ADN',
    'ADR',
    'AETR',
    'ATP',
    'CONV.INTER',
    'DIRECTIVE.CE',
    'DIRECTIVE.UE',
    'ES-TRIN',
    'PROTOCOLE',
    'REGLT',
    'REGLT.CE',
    'REGLT.CEE',
    'REGLT.D.UE',
    'REGLT.EX.UE',
    'REGLT.UE',
    'RID',
    'RPN',
    'RPNM',
    'RPNR',
    'RSI',
    'RVBR',
  ];

  List<String> _splitByCode(String text) {
    final codesPattern = _abbreviations.map(RegExp.escape).join('|');
    final tokenRegex = RegExp(
      '(ART\\.(?:[A-Z]\\.)?\\d+(?:-\\d+)*(?:\\s+AL\\.\\d+)?)|($codesPattern\\.?)',
      caseSensitive: false,
    );

    final matches = tokenRegex.allMatches(text).toList();
    final results = <String>[];
    final pending = <String>[];

    String after(int idx) {
      final start = matches[idx].end;
      final end =
          idx + 1 < matches.length ? matches[idx + 1].start : text.length;
      return text.substring(start, end);
    }

    bool isCode(String s) {
      final normalized = s.endsWith('.') ? s.substring(0, s.length - 1) : s;
      return _abbreviations.any(
        (abbr) => abbr.toUpperCase() == normalized.toUpperCase(),
      );
    }

    for (var i = 0; i < matches.length; i++) {
      final token = matches[i].group(0)!.trim();
      if (isCode(token)) {
        var trailing = after(i);
        var code = token;
        if (!token.endsWith('.')) {
          if (trailing.startsWith('.')) {
            trailing = trailing.substring(1);
          }
          code = '$token.';
        }
        for (var j = 0; j < pending.length; j++) {
          var art = pending[j].trim();
          var item = '$art $code';
          if (j == pending.length - 1) {
            item += trailing.trimRight();
          }
          results.add(item.trim());
        }
        pending.clear();
      } else {
        pending.add(token);
      }
    }

    if (pending.isNotEmpty) {
      results.addAll(pending.map((e) => e.trim()));
    }

    return results;
  }

  /// Extracts article numbers and code abbreviations from a raw string.
  /// Returns a list of `[number, abbrev]` pairs.
  List<List<String>> _extractArticleNumAndCode(String article) {
    final splitted = _splitByCode(article);
    final results = <List<String>>[];
    final codesPattern = _abbreviations.map(RegExp.escape).join('|');
    final regex = RegExp(
      'ART\\.(?:[A-Z]\\.)?(\\d+(?:-\\d+)*)(?:\\s+[§A-Z\\.0-9/,]*)*?\\s+($codesPattern)',
      caseSensitive: false,
    );

    for (final part in splitted) {
      final clean = part.replaceAll(',', '').trim();
      final match = regex.firstMatch(clean);
      if (match != null) {
        final num = match.group(1)?.replaceAll('.', '') ?? '';
        final code = match.group(2) ?? '';
        results.add([num, code]);
      }
    }

    return results;
  }

  Future<String?> _fetchLegifranceLabel(String abbrev) async {
    try {
      final resp = await _api.natinfoTexteTypeLibelleGet(abreviation: abbrev);
      if (resp.isSuccessful) {
        return resp.body?.libelleLegifranceApi ?? resp.body?.libelle;
      }
    } catch (_) {}
    return null;
  }

  Future<_ArticleInfo> _fetchArticleInfo(_ArticlePanelItem item) async {
    if (!_apiReady) return const _ArticleInfo(text: 'API non prête');
    if (!await _hasNetwork()) {
      return const _ArticleInfo(text: _offlineMessage);
    }
    final query = item.title;
    try {
      debugPrint('Loading article for query: ${Uri.encodeComponent(query)}');

      String? expectedLabel;
      String? abbr = item.codeAbbrev;
      if (abbr == null) {
        final parts = _extractArticleNumAndCode(query);
        if (parts.isNotEmpty) {
          abbr = parts.first[1];
        }
      }
      if (abbr != null) {
        expectedLabel = await _fetchLegifranceLabel(abbr);
        debugPrint('Expected code label: $expectedLabel');
      }

      final base = _api.client.baseUrl;
      final suggestUrl = Uri.parse(
        '$base/legifrance/suggest/?search_text=${Uri.encodeComponent(query)}',
      );
      final suggestResp = await http.get(suggestUrl);

      String? articleId;
      String? label;
      if (suggestResp.statusCode == 200) {
        final data = jsonDecode(suggestResp.body) as Map<String, dynamic>;
        final articles = data['results']?['articles'];
        if (articles is Map && articles.isNotEmpty) {
          for (final a in articles.values) {
            final map = a as Map;
            final l = map['label'] as String?;
            if (expectedLabel != null && l != null) {
              if (!l.toLowerCase().contains(expectedLabel.toLowerCase())) {
                continue;
              }
            }
            articleId = map['id'] as String?;
            label = l;
            break;
          }
        }
      }

      if (articleId != null) {
        final articleResp = await _api.legifranceArticleGet(
          articleId: articleId,
        );
        if (articleResp.isSuccessful && articleResp.body != null) {
          final art = articleResp.body as LegifranceArticleGet$Response;
          var html = art.texteHtml ?? art.texte ?? '';
          html = html.replaceAllMapped(
            RegExp('href="(/[^\\"]+)"'),
            (m) => 'href="https://www.legifrance.gouv.fr${m[1]}"',
          );
          return _ArticleInfo(text: html, label: label ?? query, id: articleId);
        }
        return _ArticleInfo(
          text: "Texte de l'article introuvable.",
          label: label ?? query,
          id: articleId,
        );
      }
      return _ArticleInfo(
        text:
            'Aucun article correspondant n\'a été trouvé dans les suggestions de Legifrance.',
        label: label ?? query,
        searchQuery: query,
      );
    } catch (e) {
      return _ArticleInfo(text: 'Erreur de chargement', label: query);
    }
  }

  Future<void> _launchURL(String url) async {
    final uri = Uri.parse(url);
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri, mode: LaunchMode.externalApplication);
    } else {
      if (!mounted) return;
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text("Impossible d'ouvrir le lien")),
      );
    }
  }

  Future<void> _handleExpansion(
    List<_ArticlePanelItem> list,
    int index,
    bool expanded,
  ) async {
    final item = list[index];
    setState(() {
      item.isExpanded = expanded;
    });
    if (expanded && item.text == null) {
      setState(() {
        item.isLoading = true;
        item.text = null;
        item.label = null;
        item.articleId = null;
        item.searchQuery = null;
      });
      final info = await _fetchArticleInfo(item);
      if (!mounted) return;
      setState(() {
        item.text = info.text;
        item.label = info.label;
        item.articleId = info.id;
        item.searchQuery = info.searchQuery;
        item.isLoading = false;
      });
    }
  }

  Widget _buildArticleTile(List<_ArticlePanelItem> list, int index) {
    final item = list[index];
    return Card(
      margin: const EdgeInsets.symmetric(vertical: 4),
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
      child: ExpansionTile(
        key: PageStorageKey(item.title),
        initiallyExpanded: item.isExpanded,
        onExpansionChanged: (open) => _handleExpansion(list, index, open),
        title: Text(item.title),
        children: [
          Padding(
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 8),
            child:
                item.isLoading
                    ? const Center(child: CircularProgressIndicator())
                    : item.text != null
                    ? Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Row(
                          children: [
                            Expanded(
                              child: Text(
                                item.label ?? item.title,
                                style: const TextStyle(
                                  fontWeight: FontWeight.bold,
                                ),
                              ),
                            ),
                            if (item.url != null)
                              IconButton(
                                icon: const Icon(Icons.open_in_new),
                                tooltip: 'Ouvrir sur Légifrance',
                                onPressed: () => _launchURL(item.url!),
                              ),
                          ],
                        ),
                        const SizedBox(height: 8),
                        if (item.articleId != null)
                          Html(
                            data: item.text!,
                            onLinkTap: (url, attrs, element) {
                              if (url != null) _launchURL(url);
                            },
                          )
                        else if (item.searchQuery != null)
                          Text.rich(
                            TextSpan(
                              style: Theme.of(context).textTheme.bodyMedium,
                              children: [
                                TextSpan(
                                  text:
                                      'Aucun article correspondant n\'a été trouvé dans les suggestions de Legifrance. Rechercher ',
                                ),
                                TextSpan(
                                  text: '"${item.searchQuery ?? item.title}"',
                                  style: Theme.of(context).textTheme.bodyMedium
                                      ?.copyWith(color: Colors.blue),
                                  recognizer:
                                      TapGestureRecognizer()
                                        ..onTap = () {
                                          final query =
                                              item.searchQuery ?? item.title;
                                          final url =
                                              'https://www.qwant.com/?q=${Uri.encodeComponent(query)}';
                                          _launchURL(url);
                                        },
                                ),
                                const TextSpan(text: ' sur l’Internet.'),
                              ],
                            ),
                            textAlign: TextAlign.start,
                            softWrap: true,
                            overflow: TextOverflow.visible,
                          )
                        else
                          Text(item.text ?? ''),
                      ],
                    )
                    : const SizedBox.shrink(),
          ),
        ],
      ),
    );
  }

  Widget _buildPanelList(String header, List<_ArticlePanelItem> items) {
    if (items.isEmpty) return const SizedBox.shrink();
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Padding(
          padding: const EdgeInsets.symmetric(vertical: 8),
          child: Text(
            header,
            style: const TextStyle(fontWeight: FontWeight.bold),
          ),
        ),
        ...List.generate(items.length, (i) => _buildArticleTile(items, i)),
      ],
    );
  }

  Future<void> _copyText(String text) async {
    await Clipboard.setData(ClipboardData(text: text));
    ScaffoldMessenger.of(context).showSnackBar(
      const SnackBar(content: Text('Texte copié dans le presse-papiers')),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(title: const Text('Textes juridiques')),
      body:
          !_apiReady || _loading
              ? const Center(child: CircularProgressIndicator())
              : _offline
              ? Center(
                child: Padding(
                  padding: const EdgeInsets.all(16),
                  child: Column(
                    mainAxisAlignment: MainAxisAlignment.center,
                    children: [
                      Text(_offlineMessage, textAlign: TextAlign.center),
                      const SizedBox(height: 16),
                      ElevatedButton(
                        onPressed: _checkConnectivityAndLoad,
                        child: const Text('Réessayer'),
                      ),
                    ],
                  ),
                ),
              )
              : Padding(
                padding: const EdgeInsets.all(16),
                child: ListView(
                  children: [
                    _buildPanelList('Articles « Définie par »', _definieItems),
                    _buildPanelList(
                      'Articles « Réprimée par »',
                      _reprimeeItems,
                    ),
                    const SizedBox(height: 16),
                    if (widget.natinf.definiePar != null &&
                        widget.natinf.definiePar!.trim().isNotEmpty)
                      Padding(
                        padding: const EdgeInsets.only(bottom: 8.0),
                        child: GestureDetector(
                          onLongPress:
                              () => _copyText(widget.natinf.definiePar!),
                          child: Text(
                            'Définie par : ${widget.natinf.definiePar}',
                          ),
                        ),
                      ),
                    if (widget.natinf.reprimeePar != null &&
                        widget.natinf.reprimeePar!.trim().isNotEmpty)
                      Padding(
                        padding: const EdgeInsets.only(bottom: 16.0),
                        child: GestureDetector(
                          onLongPress:
                              () => _copyText(widget.natinf.reprimeePar!),
                          child: Text(
                            'Réprimée par : ${widget.natinf.reprimeePar}',
                          ),
                        ),
                      ),
                  ],
                ),
              ),
    );
  }
}
