import 'dart:async';
import 'dart:convert';
import 'dart:io';

import 'package:http/http.dart' as http;
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_download_exception.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_cache_service.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_mappers.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';

class NatinfRemoteService {
  NatinfRemoteService({
    required Swagger apiClient,
    required Swagger categoriesApiClient,
    required NatinfCacheService cacheService,
    required http.Client httpClient,
    SourceLogger? logger,
  }) : _apiClient = apiClient,
       _categoriesApiClient = categoriesApiClient,
       _cacheService = cacheService,
       _httpClient = httpClient,
       _logger = logger ?? SourceLogger(tag: 'NATINF');

  final Swagger _apiClient;
  final Swagger _categoriesApiClient;
  final NatinfCacheService _cacheService;
  final http.Client _httpClient;
  final SourceLogger _logger;
  static const Duration _categoriesTimeout = Duration(seconds: 90);

  Future<List<CategoryBrief>> fetchRootCategories() async {
    try {
      final response = await _categoriesApiClient.categoriesGet().timeout(
        _categoriesTimeout,
      );
      final categories = response.body;
      if (response.isSuccessful && categories != null) {
        await _cacheService.saveCategories(categories);
        return categories
            .where((c) => c.parent == null)
            .map((c) => CategoryBrief(id: c.id, name: c.name))
            .toList();
      }
      _logger.warn('Categories endpoint returned ${response.statusCode}');
    } catch (error, stackTrace) {
      _logger.warn(
        'Unable to fetch categories from network: $error\n$stackTrace',
      );
    }

    final cached = await _cacheService.getCachedRootCategories();
    if (cached.isNotEmpty) {
      return cached;
    }
    throw Exception('Impossible de charger les catégories');
  }

  Future<Category> fetchCategory(int id) async {
    try {
      final response = await _categoriesApiClient
          .categoriesCategoryIdGet(categoryId: id.toString())
          .timeout(const Duration(seconds: 180));
      final body = response.body;
      if (response.isSuccessful && body != null) {
        final natinfs = materializeNatinfs(body.natinfs);
        final category = body.copyWith(natinfs: natinfs);
        await _cacheService.saveCategory(category);
        return category;
      }
    } on TimeoutException catch (error, stackTrace) {
      _logger.warn('Category $id fetch timeout: $error\n$stackTrace');
    } catch (error, stackTrace) {
      _logger.warn('Category $id fetch failed: $error\n$stackTrace');
    }

    final local = await _cacheService.getCachedCategory(id);
    if (local != null) {
      return local;
    }
    throw Exception('Impossible de charger la catégorie');
  }

  Future<void> fetchInitialData({
    void Function(String)? onStatus,
    void Function(double)? onProgress,
    void Function(int)? onPhase,
  }) async {
    onStatus?.call("Téléchargement des données depuis l'API...");
    onPhase?.call(1);
    final base = _apiClient.client.baseUrl;
    final uri = Uri.parse('${base.toString()}/natinfo/');
    _logger.info('Starting initial NATINF download from $uri');
    final http.Response httpResponse;
    try {
      httpResponse = await _httpClient
          .get(uri)
          .timeout(const Duration(seconds: 60));
    } on TimeoutException catch (error, stackTrace) {
      _logger.warn('Initial download timed out: $error\n$stackTrace');
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.network,
        message:
            "Timeout (60s) lors de la récupération des données NATINF depuis $uri.",
        cause: error,
        stackTrace: stackTrace,
      );
    } on SocketException catch (error, stackTrace) {
      _logger.warn(
        'Network error during initial download: $error\n$stackTrace',
      );
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.network,
        message:
            "Connexion réseau indisponible pour télécharger les données NATINF.",
        cause: error,
        stackTrace: stackTrace,
      );
    } catch (error, stackTrace) {
      _logger.warn(
        'Unexpected network error during initial download: $error\n$stackTrace',
      );
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.unknown,
        message: "Erreur réseau inattendue lors du téléchargement initial.",
        cause: error,
        stackTrace: stackTrace,
      );
    }

    if (httpResponse.statusCode != 200) {
      final message =
          "Le serveur a répondu ${httpResponse.statusCode} lors du téléchargement initial.";
      _logger.warn('$message BodyLength=${httpResponse.bodyBytes.length}');
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.network,
        message: message,
      );
    }
    if (httpResponse.body.isEmpty) {
      const message =
          "Réponse vide lors du téléchargement initial des données NATINF.";
      _logger.warn(message);
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.parsing,
        message: message,
      );
    }

    late List<dynamic> rawList;
    try {
      rawList = _decodeInitialPayload(httpResponse.body);
    } on FormatException catch (error, stackTrace) {
      _logger.warn('Unable to decode NATINF payload: $error\n$stackTrace');
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.parsing,
        message:
            "Les données téléchargées ont un format inattendu (JSON invalide).",
        cause: error,
        stackTrace: stackTrace,
      );
    }

    final List<Natinf> list = [];
    try {
      for (final item in rawList) {
        final parsed = parseServerNatinf(
          Map<String, dynamic>.from(item as Map),
        );
        list.add(parsed.natinf);
      }
    } catch (error, stackTrace) {
      _logger.warn('Failed to parse NATINF entries: $error\n$stackTrace');
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.parsing,
        message:
            "Les données NATINF téléchargées ne peuvent pas être interprétées.",
        cause: error,
        stackTrace: stackTrace,
      );
    }

    onStatus?.call(
      "Téléchargement terminé. Importation des données dans la base locale...",
    );
    onPhase?.call(2);
    final total = list.length;
    final stopwatch = Stopwatch()..start();
    try {
      for (int i = 0; i < total; i++) {
        final natinf = list[i];
        await _cacheService.saveNatinf(natinf);
        final progressFraction = (i + 1) / total;
        final progressPercent = (progressFraction * 100).toStringAsFixed(0);
        final elapsed = stopwatch.elapsed.inSeconds;
        final speed =
            elapsed > 0 ? ((i + 1) / elapsed).toStringAsFixed(1) : "calcul...";
        onStatus?.call(
          "Importation: $progressPercent% (${i + 1}/$total) - $speed items/s",
        );
        onProgress?.call(progressFraction);
      }
    } catch (error, stackTrace) {
      _logger.warn('Failed to persist NATINF payload: $error\n$stackTrace');
      throw NatinfDownloadException(
        type: NatinfDownloadIssueType.storage,
        message: "Impossible d'écrire les données NATINF en local (stockage).",
        cause: error,
        stackTrace: stackTrace,
      );
    }
    onStatus?.call("Importation terminée.");
  }

  Future<List<Natinf>> searchNatinf(
    String query, {
    void Function(String)? onStatus,
  }) async {
    onStatus?.call("Envoi de la requête API NATINFo...");
    await _cacheService.clearDocsProUnauthorized();
    final base = _apiClient.client.baseUrl;
    final uri = Uri.parse(
      '${base.toString()}/natinfo/search/',
    ).replace(queryParameters: {'q': query});
    final response = await _httpClient
        .get(uri)
        .timeout(const Duration(seconds: 10));
    onStatus?.call("Réponse API reçue: ${response.statusCode}");
    if (response.statusCode == 200 && response.body.isNotEmpty) {
      final decoded = jsonDecode(response.body);
      final List<dynamic> rawList;
      if (decoded is List) {
        rawList = decoded;
      } else if (decoded is Map && decoded['results'] is List) {
        rawList = decoded['results'] as List;
      } else {
        throw Exception('Format de réponse inattendu: ${decoded.runtimeType}');
      }
      final List<Natinf> list = [];
      for (final item in rawList) {
        final parsed = parseServerNatinf(
          Map<String, dynamic>.from(item as Map),
        );
        final numero = parsed.numero;
        if (numero != null) {
          await _cacheService.updateDocsProInfo(
            numero,
            unauthorized: parsed.docsProUnauthorized,
            available: parsed.docsProAvailable,
          );
          if (parsed.references.isNotEmpty) {
            await _cacheService.saveReferences(numero, parsed.references);
          }
        }
        list.add(parsed.natinf);
      }
      if (list.isNotEmpty) {
        onStatus?.call("Réponse API valide, mise à jour BD locale...");
        for (final natinf in list) {
          await _cacheService.saveNatinf(natinf);
        }
        return list;
      } else {
        throw Exception("Aucune NATINF trouvée sur le serveur.");
      }
    } else {
      throw Exception("Erreur serveur : ${response.statusCode}");
    }
  }

  Future<List<NatinfDocPro>> fetchDocPro(String numero) async {
    try {
      final response = await _apiClient.natinfoDocProGet(numeroNatinf: numero);
      if (response.isSuccessful && response.body != null) {
        final docs = response.body!
            .where((doc) => doc.url.isNotEmpty)
            .toList(growable: false);
        if (docs.isNotEmpty) {
          return docs;
        }
      }
      return const [];
    } catch (error, stackTrace) {
      _logger.warn('Failed to fetch DocPro for $numero: $error\n$stackTrace');
      rethrow;
    }
  }

  List<dynamic> _decodeInitialPayload(String body) {
    final decoded = jsonDecode(body);
    if (decoded is List) {
      return decoded;
    }
    if (decoded is Map && decoded['results'] is List) {
      return decoded['results'] as List<dynamic>;
    }
    throw FormatException('Unexpected payload type: ${decoded.runtimeType}');
  }
}
