import 'package:flutter/material.dart';

/// A reusable download screen widget that displays progress and status,
/// and shows retry and issue handling options.
class DownloadScreen extends StatelessWidget {
  final String status;
  final double progress;
  final int phase;
  final VoidCallback onRetry;
  final VoidCallback? onIssueTap;
  final VoidCallback? onSkip;

  const DownloadScreen({
    super.key,
    required this.status,
    required this.progress,
    required this.phase,
    required this.onRetry,
    this.onIssueTap,
    this.onSkip,
  });

  @override
  Widget build(BuildContext context) {
    final bool hasError =
        status.startsWith("Erreur") ||
        status.startsWith("Téléchargement annulé");
    Widget content;
    if (hasError) {
      content = Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          Icon(Icons.error, size: 48, color: Colors.red.shade300),
          const SizedBox(height: 16),
          Text(
            status,
            textAlign: TextAlign.center,
            style: const TextStyle(fontSize: 16),
          ),
          const SizedBox(height: 16),
          ElevatedButton(onPressed: onRetry, child: const Text("Réessayer")),
          if (onSkip != null) ...[
            const SizedBox(height: 8),
            TextButton(
              onPressed: onSkip,
              child: const Text("Continuer avec la recherche en ligne"),
            ),
          ],
          if (onIssueTap != null) ...[
            const SizedBox(height: 16),
            GestureDetector(
              onTap: onIssueTap,
              child: Text(
                "Ouvrir une issue sur Codeberg",
                style: TextStyle(
                  decoration: TextDecoration.underline,
                  color: Theme.of(context).colorScheme.secondary,
                ),
              ),
            ),
          ],
        ],
      );
    } else {
      Widget progressWidget;
      if (phase == 1) {
        progressWidget = const CircularProgressIndicator();
      } else if (phase == 2) {
        progressWidget = LinearProgressIndicator(
          value: progress,
          minHeight: 8,
          backgroundColor: Colors.grey.shade300,
          valueColor: AlwaysStoppedAnimation<Color>(
            Theme.of(context).primaryColor,
          ),
        );
      } else {
        progressWidget = const CircularProgressIndicator();
      }
      content = Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          progressWidget,
          const SizedBox(height: 16),
          Text(status, textAlign: TextAlign.center),
        ],
      );
    }

    return Scaffold(
      body: Center(
        child: Padding(
          padding: const EdgeInsets.symmetric(horizontal: 24),
          child: content,
        ),
      ),
      bottomNavigationBar: Padding(
        padding: const EdgeInsets.symmetric(vertical: 8.0, horizontal: 16.0),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            Text(
              "NATINFo+ - Développé par retiolus",
              textAlign: TextAlign.center,
              style: TextStyle(
                fontSize: 12,
                color: Colors.grey.shade700,
                fontStyle: FontStyle.italic,
              ),
            ),
            const SizedBox(height: 4),
            Text(
              "Cette application n'est pas un service officiel de l'État français et n'est ni affiliée, ni soutenue par l'administration publique.",
              textAlign: TextAlign.center,
              style: TextStyle(
                fontSize: 12,
                color: Colors.grey.shade600,
                fontStyle: FontStyle.italic,
              ),
            ),
          ],
        ),
      ),
    );
  }
}
