import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import 'package:url_launcher/url_launcher.dart';

import '../domain/auth_session.dart';
import '../presentation/auth_provider.dart';

/// Displays current account details, token validity and quick actions.
class AccountPage extends StatelessWidget {
  /// Creates an account page that surfaces the current session state.
  const AccountPage({super.key});

  @override
  Widget build(BuildContext context) {
    final auth = context.watch<AuthProvider>();
    final session = auth.session;
    return Scaffold(
      appBar: AppBar(title: const Text('Mon compte')),
      body:
          session == null
              ? const _DisconnectedState()
              : _ConnectedState(session: session, provider: auth),
    );
  }
}

class _DisconnectedState extends StatelessWidget {
  const _DisconnectedState();

  @override
  Widget build(BuildContext context) {
    return Center(
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          const Text('Aucune session active.'),
          const SizedBox(height: 12),
          ElevatedButton(
            onPressed: () => Navigator.of(context).pop(),
            child: const Text('Retour'),
          ),
        ],
      ),
    );
  }
}

class _ConnectedState extends StatelessWidget {
  const _ConnectedState({required this.session, required this.provider});

  final AuthProvider provider;
  final AuthSession session;

  @override
  Widget build(BuildContext context) {
    return SafeArea(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          children: [
            Expanded(
              child: ListView(
                children: [
                  _SectionCard(
                    title: 'Profil',
                    children: [
                      _InfoLine(
                        label: 'Utilisateur',
                        value: session.username ?? 'Inconnu',
                      ),
                      if (session.tokenLabel != null &&
                          session.tokenLabel!.isNotEmpty)
                        _InfoLine(
                          label: 'Libellé du jeton',
                          value: session.tokenLabel!,
                        ),
                    ],
                  ),
                  const SizedBox(height: 12),
                  _DocProCard(hasDocPro: session.hasDocPro ?? false),
                  const SizedBox(height: 12),
                  _TokenSection(session: session),
                ],
              ),
            ),
            const SizedBox(height: 16),
            _ActionButtons(provider: provider),
          ],
        ),
      ),
    );
  }
}

class _TokenSection extends StatelessWidget {
  const _TokenSection({required this.session});

  final AuthSession session;

  @override
  Widget build(BuildContext context) {
    return _SectionCard(
      title: 'Statut des jetons',
      children: [
        _InfoLine(
          label: 'Expiration accès',
          value: _formatDate(context, session.accessExpiresAt),
        ),
        _InfoLine(
          label: 'Expiration refresh',
          value: _formatDate(context, session.refreshExpiresAt),
        ),
        _InfoLine(
          label: 'Dernier refresh',
          value: _formatDate(context, session.lastRefresh),
        ),
      ],
    );
  }
}

class _DocProCard extends StatelessWidget {
  const _DocProCard({required this.hasDocPro});

  final bool hasDocPro;

  @override
  Widget build(BuildContext context) {
    final colorScheme = Theme.of(context).colorScheme;
    final backgroundColor =
        hasDocPro
            ? colorScheme.secondaryContainer
            : colorScheme.surfaceContainerHighest;
    final foregroundColor =
        hasDocPro ? colorScheme.onSecondaryContainer : colorScheme.onSurface;
    final icon = hasDocPro ? Icons.badge_rounded : Icons.shield_outlined;
    final helper =
        hasDocPro ? 'Accès DocPro actif.' : 'DocPro non associé à ce compte.';

    return Card(
      child: Padding(
        padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
        child: Row(
          children: [
            CircleAvatar(
              radius: 22,
              backgroundColor: backgroundColor,
              child: Icon(icon, color: foregroundColor),
            ),
            const SizedBox(width: 12),
            Expanded(
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    'DocPro',
                    style: Theme.of(context).textTheme.titleMedium,
                  ),
                  const SizedBox(height: 4),
                  Text(helper, style: Theme.of(context).textTheme.bodyMedium),
                ],
              ),
            ),
            const SizedBox(width: 12),
            Chip(
              label: Text(hasDocPro ? 'Actif' : 'Inactif'),
              backgroundColor: backgroundColor,
              labelStyle: TextStyle(color: foregroundColor),
            ),
          ],
        ),
      ),
    );
  }
}

class _SectionCard extends StatelessWidget {
  const _SectionCard({required this.title, required this.children});

  final String title;
  final List<Widget> children;

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(title, style: theme.textTheme.titleMedium),
            const SizedBox(height: 12),
            ..._withSpacing(children),
          ],
        ),
      ),
    );
  }

  List<Widget> _withSpacing(List<Widget> items) {
    final spaced = <Widget>[];
    for (var i = 0; i < items.length; i++) {
      spaced.add(items[i]);
      if (i != items.length - 1) {
        spaced.add(const SizedBox(height: 12));
        spaced.add(const Divider(height: 1));
        spaced.add(const SizedBox(height: 12));
      }
    }
    return spaced;
  }
}

class _InfoLine extends StatelessWidget {
  const _InfoLine({required this.label, required this.value});

  final String label;
  final String value;

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          label,
          style: theme.textTheme.labelMedium?.copyWith(
            color: theme.colorScheme.primary,
          ),
        ),
        const SizedBox(height: 4),
        Text(value, style: theme.textTheme.bodyLarge),
      ],
    );
  }
}

class _ActionButtons extends StatelessWidget {
  const _ActionButtons({required this.provider});

  final AuthProvider provider;
  static final Uri _onlineAccountUri = Uri.https('natinfo.app', '/account');

  Future<void> _openOnlineAccount(BuildContext context) async {
    final launched = await launchUrl(
      _onlineAccountUri,
      mode: LaunchMode.externalApplication,
    );
    if (!launched && context.mounted) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(
          content: Text('Impossible d\'ouvrir le compte en ligne.'),
        ),
      );
    }
  }

  @override
  Widget build(BuildContext context) {
    return Row(
      children: [
        Expanded(
          child: ElevatedButton.icon(
            onPressed:
                provider.isLoading
                    ? null
                    : () async {
                      await provider.refresh();
                      if (context.mounted) {
                        ScaffoldMessenger.of(context).showSnackBar(
                          const SnackBar(content: Text('Jeton rafraîchi')),
                        );
                      }
                    },
            icon: const Icon(Icons.refresh),
            label: const Text('Rafraîchir'),
          ),
        ),
        const SizedBox(width: 12),
        Expanded(
          child: OutlinedButton.icon(
            onPressed:
                provider.isLoading
                    ? null
                    : () async {
                      await provider.logout();
                      if (context.mounted) {
                        Navigator.of(context).pop();
                      }
                    },
            icon: const Icon(Icons.logout),
            label: const Text('Déconnexion'),
          ),
        ),
        const SizedBox(width: 12),
        Expanded(
          child: OutlinedButton.icon(
            onPressed:
                provider.isLoading ? null : () => _openOnlineAccount(context),
            icon: const Icon(Icons.open_in_new),
            label: const Text('Compte en ligne'),
          ),
        ),
      ],
    );
  }
}

String _formatDate(BuildContext context, DateTime? dateTime) {
  if (dateTime == null) {
    return 'Non rafraîchi';
  }
  final local = dateTime.toLocal();
  final localizations = MaterialLocalizations.of(context);
  final date = localizations.formatMediumDate(local);
  final time = localizations.formatTimeOfDay(
    TimeOfDay.fromDateTime(local),
    alwaysUse24HourFormat: true,
  );
  return '$date • $time';
}
