import 'dart:typed_data';

import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/shared/data_sources/source_environment.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';
import 'package:natinfo_flutter/shared/data_sources/source_override_store.dart';
import 'package:natinfo_flutter/shared/data_sources/source_preferences.dart';
import 'package:shared_preferences/shared_preferences.dart';

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  const specJson = '''
  {
    "sources": [
      {
        "id": "demo-source",
        "dataset": "demo",
        "name": "Demo dataset",
        "type": "asset",
        "uri": "asset://demo.json",
        "scope": "demo",
        "priority": 1,
        "requiredAtBuild": false,
        "requiresNetwork": false,
        "schemaVersion": 1,
        "checksum": null,
        "checksumAlgo": null
      }
    ]
  }
  ''';

  group('SourceEnvironmentLoader', () {
    setUp(() {
      SharedPreferences.setMockInitialValues({});
    });

    test('loads overrides from prefs and builds registry/resolver', () async {
      SharedPreferences.setMockInitialValues({
        SourceOverrideStore.keyFor('demo'): 'https://custom.test',
      });

      final env = await SourceEnvironmentLoader.load(
        bundle: _FakeAssetBundle({kSourceRegistryAsset: specJson}),
      );

      expect(env.preferences, isA<SharedPreferences>());
      expect(env.overrides['demo']?.toString(), 'https://custom.test');
      expect(env.registry.all, isNotEmpty);
      expect(env.resolver.resolve('demo').id, 'custom-demo');
    });

    test('honors provided collaborators', () async {
      final prefs = await SharedPreferences.getInstance();
      final registry = SourceRegistry([
        SourceSpec(
          id: 'provided',
          dataset: 'demo',
          name: 'Demo dataset',
          type: SourceType.file,
          uri: Uri.parse('file:///tmp/demo'),
          scope: 'demo',
          priority: 1,
          requiredAtBuild: false,
          requiresNetwork: false,
          schemaVersion: 1,
        ),
      ]);
      final resolver = SourceResolver(registry);
      final sourcePrefs = SourcePreferences(prefs, globalNetworkAllowed: false);
      final overrides = {'demo': Uri.parse('https://ignored.test')};

      final env = await SourceEnvironmentLoader.load(
        preferences: prefs,
        registry: registry,
        resolver: resolver,
        sourcePreferences: sourcePrefs,
        overrides: overrides,
      );

      expect(identical(env.preferences, prefs), isTrue);
      expect(identical(env.registry, registry), isTrue);
      expect(identical(env.resolver, resolver), isTrue);
      expect(identical(env.sourcePreferences, sourcePrefs), isTrue);
      expect(env.overrides, equals(overrides));
    });
  });
}

class _FakeAssetBundle extends CachingAssetBundle {
  _FakeAssetBundle(this.assets);

  final Map<String, String> assets;

  @override
  Future<ByteData> load(String key) async {
    final value = _read(key);
    return ByteData.view(Uint8List.fromList(value.codeUnits).buffer);
  }

  @override
  Future<String> loadString(String key, {bool cache = true}) async {
    return _read(key);
  }

  String _read(String key) {
    final value = assets[key];
    if (value == null) throw FlutterError('Asset not found: $key');
    return value;
  }
}
