import 'dart:io';
import 'dart:typed_data';

import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/shared/data_sources/http_source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

void main() {
  group('HttpSourceAdapter', () {
    test('loads bytes and validates checksum', () async {
      final client = MockClient((request) async {
        return http.Response.bytes(Uint8List.fromList([1, 2, 3]), 200);
      });
      final adapter = HttpSourceAdapter(client: client);
      final spec = SourceSpec(
        id: 'api',
        dataset: 'natinf',
        name: 'Infractions NATINFo',
        type: SourceType.api,
        uri: Uri.parse('https://example.com'),
        scope: 'natinf',
        priority: 1,
        requiredAtBuild: false,
        requiresNetwork: true,
        schemaVersion: 1,
        checksum: '5289df737df57326fcdd22597afb1fac',
        checksumAlgo: 'md5',
      );

      final result = await adapter.load(spec);
      expect(result.bytes, [1, 2, 3]);
      expect(result.integrity.isValid, isTrue);
    });

    test('returns mismatch status when checksum differs', () async {
      final client = MockClient((request) async {
        return http.Response('payload', 200);
      });
      final adapter = HttpSourceAdapter(client: client);
      final spec = SourceSpec(
        id: 'api',
        dataset: 'natinf',
        name: 'Infractions NATINFo',
        type: SourceType.api,
        uri: Uri.parse('https://example.com'),
        scope: 'natinf',
        priority: 1,
        requiredAtBuild: false,
        requiresNetwork: true,
        schemaVersion: 1,
        checksum: 'baduuid',
        checksumAlgo: 'sha1',
      );

      final result = await adapter.load(spec);
      expect(result.integrity.status, ChecksumStatus.mismatch);
    });

    test('throws on HTTP error', () async {
      final client = MockClient((request) async {
        return http.Response('error', 500);
      });
      final adapter = HttpSourceAdapter(client: client);
      final spec = SourceSpec(
        id: 'api',
        dataset: 'natinf',
        name: 'Infractions NATINFo',
        type: SourceType.api,
        uri: Uri.parse('https://example.com'),
        scope: 'natinf',
        priority: 1,
        requiredAtBuild: false,
        requiresNetwork: true,
        schemaVersion: 1,
      );

      expect(() async => adapter.load(spec), throwsA(isA<HttpException>()));
    });
  });
}
