import 'package:flutter/material.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';

enum RacePresumee { amstaff, tosa, rottweiler, mastiffBoerbull, inconnu }

enum LOFStatus { lof, nonLof, inconnu }

enum CatResult { cat1, cat2, nonCat, indetermine }

enum AgeBand { mois0a7, mois8a12, plusDe12 }

enum Couleur { noirFeu, fauveRouge, autre }

enum Museau { court, moyen, long }

enum StopTete { peuMarque, moyen, prononce }

enum PeauCoussinets { peauLacheDewlap, peauFerme }

enum Corpulence { massif, puissantCompact, plutotSec }

enum Oreilles { roseSemiDressees, tombantesTriangulaires, autres }

enum Poil { rasCourt, autre }

enum ADejaMordu { oui, non }

class ChiensDangereuxPage extends StatefulWidget {
  const ChiensDangereuxPage({super.key});
  @override
  State<ChiensDangereuxPage> createState() => _ChiensDangereuxPageState();
}

class _ChiensDangereuxPageState extends State<ChiensDangereuxPage> {
  // Étape 1 – infos rapides
  RacePresumee _race = RacePresumee.inconnu;
  LOFStatus _lof = LOFStatus.inconnu;
  AgeBand _age = AgeBand.plusDe12;
  ADejaMordu _mordu = ADejaMordu.non;

  // Étape 1 bis – morphologie (si race/LOF incertains)
  double? _hauteurCm; // garrot
  double? _poidsKg;
  Couleur _couleur = Couleur.autre;
  Museau _museau = Museau.moyen;
  StopTete _stop = StopTete.moyen;
  PeauCoussinets _peau = PeauCoussinets.peauFerme;
  Corpulence _corpulence = Corpulence.puissantCompact;
  Oreilles _oreilles = Oreilles.autres;
  Poil _poil = Poil.rasCourt;

  // Résultats
  CatResult? _result;
  String _explication = '';
  List<String> _obligDetenteur = [];
  List<String> _obligPromeneur = [];
  List<String> _obligChien = [];
  List<String> _pouvoirsMaire = [];
  List<String> _notes = [];

  @override
  void initState() {
    super.initState();
    MatomoService().trackPage(
      title: 'Chiens dangereux',
      path: '/tools/chiens/',
    );
  }

  // ---------- Helpers ----------
  double? _parse(String t) {
    if (t.trim().isEmpty) return null;
    return double.tryParse(t.replaceAll(',', '.'));
  }

  InputDecoration _dec(String label, {String? hint}) => InputDecoration(
    labelText: label,
    hintText: hint,
    border: const OutlineInputBorder(),
    contentPadding: const EdgeInsets.symmetric(horizontal: 12, vertical: 10),
  );

  Future<void> _open(String url) async {
    final ok = await launchUrl(
      Uri.parse(url),
      mode: LaunchMode.externalApplication,
    );
    if (!ok && mounted) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text("Impossible d’ouvrir le lien")),
      );
    }
  }

  // ---------- Mini parseur **gras** → RichText ----------
  InlineSpan _span(String text, {bool bold = false}) => TextSpan(
    text: text,
    style: TextStyle(
      fontWeight: bold ? FontWeight.w600 : FontWeight.normal,
      height: 1.25,
      color: Theme.of(context).textTheme.bodyMedium?.color,
    ),
  );

  /// Convertit "xxx **GRAS** yyy" en TextSpan
  Text _md(String s, {TextStyle? style}) {
    final reg = RegExp(r'\*\*(.+?)\*\*');
    final spans = <InlineSpan>[];
    int index = 0;
    for (final m in reg.allMatches(s)) {
      if (m.start > index) spans.add(_span(s.substring(index, m.start)));
      spans.add(_span(m.group(1)!, bold: true));
      index = m.end;
    }
    if (index < s.length) spans.add(_span(s.substring(index)));
    return Text.rich(TextSpan(children: spans), style: style);
  }

  // ---------- Cœur : qualification ----------
  void _qualifier() {
    // 1) Règles juridiques simples (race/LOF)
    // - Cat. 1 : chiens NON LOF de type American Staffordshire terrier (« pit-bull »),
    //            type Mastiff/Boerboel (« boerbull »), type Tosa.
    // - Cat. 2 : American Staffordshire terrier LOF, Tosa LOF, Rottweiler (LOF ou assimilable).
    // - Sinon : non catégorisé (mais un chien mordeur aura des mesures).
    CatResult res = CatResult.indetermine;
    String why = '';

    if (_race == RacePresumee.amstaff) {
      if (_lof == LOFStatus.lof) {
        res = CatResult.cat2;
        why =
            "American Staffordshire terrier (dit « amstaff ») inscrit au LOF → 2ᵉ catégorie (chien de garde/défense).";
      } else if (_lof == LOFStatus.nonLof) {
        res = CatResult.cat1;
        why =
            "Type American Staffordshire terrier non LOF (souvent appelé « pit-bull ») → 1ʳᵉ catégorie (chiens d’attaque).";
      }
    } else if (_race == RacePresumee.tosa) {
      if (_lof == LOFStatus.lof) {
        res = CatResult.cat2;
        why = "Tosa (chien de combat japonais) inscrit au LOF → 2ᵉ catégorie.";
      } else if (_lof == LOFStatus.nonLof) {
        res = CatResult.cat1;
        why = "Type Tosa non LOF → 1ʳᵉ catégorie.";
      }
    } else if (_race == RacePresumee.rottweiler) {
      // Rottweiler : LOF ou non LOF assimilable → toujours 2e cat.
      res = CatResult.cat2;
      why =
          "Rottweiler (souvent « rott ») – LOF ou morphologiquement assimilable → 2ᵉ catégorie.";
    } else if (_race == RacePresumee.mastiffBoerbull) {
      if (_lof == LOFStatus.nonLof || _lof == LOFStatus.inconnu) {
        res = CatResult.cat1;
        why = "Type Mastiff / Boerboel (« boerbull ») non LOF → 1ʳᵉ catégorie.";
      }
    }

    // 2) Si toujours indéterminé → petit scoring morphologique
    if (res == CatResult.indetermine) {
      final sAmstaff = _scoreAmstaff();
      final sMastiff = _scoreMastiffBoerbull();
      final sTosa = _scoreTosa();
      final sRott = _scoreRott();

      final scores = {
        'amstaffType': sAmstaff,
        'mastiffType': sMastiff,
        'tosaType': sTosa,
        'rottType': sRott,
      };

      final best = scores.entries.reduce((a, b) => a.value >= b.value ? a : b);

      if (best.value >= 60) {
        switch (best.key) {
          case 'amstaffType':
            res = CatResult.cat1;
            why =
                "Morphologie proche d’un type American Staffordshire non LOF (score $sAmstaff).";
            break;
          case 'mastiffType':
            res = CatResult.cat1;
            why =
                "Morphologie proche d’un type Mastiff/Boerboel non LOF (score $sMastiff).";
            break;
          case 'tosaType':
            res = CatResult.cat1;
            why = "Morphologie proche d’un type Tosa non LOF (score $sTosa).";
            break;
          case 'rottType':
            res = CatResult.cat2;
            why =
                "Morphologie proche du Rottweiler (LOF ou assimilable) (score $sRott).";
            break;
        }
      } else {
        res = CatResult.nonCat;
        why =
            "Aucun signal morphologique fort (scores faibles) → a priori non catégorisé.";
      }
    }

    // 3) Chien mordeur : on le marque pour les obligations spécifiques
    final mordeur = _mordu == ADejaMordu.oui;

    // 4) Construire les obligations/pouvoirs
    final oblDet = <String>[];
    final oblDog = <String>[];
    final maire = <String>[];
    final notes = <String>[];

    // Commun cat. 1 & 2
    if (res == CatResult.cat1 || res == CatResult.cat2) {
      oblDet.addAll([
        "Obtenir un **permis de détention** (provisoire si le chien a entre 8 et 12 mois, puis **définitif** quand toutes les pièces sont réunies).",
        "Suivre la **formation de 7 h** (attestation d’aptitude).",
        "Faire pratiquer l’**évaluation comportementale** (entre **8 et 12 mois**, puis selon le **niveau de risque** fixé par le vétérinaire).",
        "Souscrire une **assurance responsabilité civile** couvrant le chien.",
        "Faire **identifier** le chien (I-CAD) et veiller à la **vaccination antirabique** si requise localement.",
        "Présenter **permis** et **justificatifs** à tout contrôle.",
      ]);
      oblDog.addAll([
        "En **espace public** et dans les **transports** : **laisse courte** tenue par une personne **majeure** et **muselière**.",
      ]);
      notes.add(
        "Personnes **interdites** : mineurs ; majeurs sous tutelle (sauf autorisation du juge) ; personnes condamnées pour certains crimes/délits ; personne à qui un chien a déjà été retiré.",
      );
    }

    // Spécifique catégorie 1
    if (res == CatResult.cat1) {
      oblDet.insertAll(0, [
        "**Acquisition, cession, importation et introduction interdites.**",
        "**Stérilisation obligatoire** (certificat vétérinaire).",
      ]);
      oblDog.addAll([
        "Accès **interdit** aux **lieux publics** (hors voie publique), **locaux ouverts au public** et **transports**.",
        "Sur **voie publique** : **laisse + muselière** obligatoires.",
      ]);
    }

    // Spécifique catégorie 2
    if (res == CatResult.cat2) {
      oblDog.add(
        "Accès aux lieux publics **possible** sous **laisse + muselière** (et selon **arrêtés municipaux** éventuels).",
      );
    }

    // Non catégorisé : “circonstances” + morsure
    if (res == CatResult.nonCat) {
      notes.add(
        "Même non catégorisé, un chien peut être jugé **dangereux eu égard aux circonstances** : le **maire** peut imposer muselière/laisse, évaluation, voire **mesures plus strictes**.",
      );
    }

    // Chien mordeur (toutes catégories)
    if (mordeur) {
      oblDet.add(
        "En cas de **morsure**, **déclaration en mairie** et **évaluation comportementale**. **Surveillance sanitaire** (rage) selon protocole vétérinaire.",
      );
    }

    // Pouvoirs du maire (tous cas)
    maire.addAll([
      "Instruit les demandes de **permis** (peut **refuser** ou **retirer** le permis selon l’évaluation).",
      "Peut imposer des **mesures de police** (muselière, laisse, évaluation, stérilisation cat. 1, etc.) et prendre des **arrêtés** adaptés aux lieux.",
      "En cas de **danger grave et immédiat** : **capture**, **mise en fourrière** ; en ultime recours, **euthanasie** (après avis vétérinaire et selon la loi).",
    ]);

    setState(() {
      _result = res;
      _explication = why;
      _obligDetenteur = oblDet;
      _obligPromeneur = [
        if (res == CatResult.cat1 || res == CatResult.cat2)
          "Être **majeur**, tenir le chien en **laisse courte** et lui mettre une **muselière** en espace public et transports ; avoir les **documents** (permis, assurance, identification) disponibles.",
        if (res == CatResult.cat1)
          "Ne pas entrer avec le chien dans les **lieux publics** (hors voie publique) ni **transports**.",
        if (res == CatResult.nonCat)
          "Respecter toute **obligation locale** (arrêté du maire) ; muselière/laisse si imposées.",
      ];
      _obligChien = oblDog;
      _pouvoirsMaire = maire;
      _notes = notes;
    });
  }

  // ---------- Scorings morphologiques (0–100) ----------
  int _scoreAmstaff() {
    int s = 0;
    if (_poil == Poil.rasCourt) s += 10;
    if (_corpulence == Corpulence.puissantCompact) s += 20;
    if (_stop == StopTete.prononce) s += 15;
    if (_museau == Museau.moyen) s += 10;
    if (_oreilles == Oreilles.roseSemiDressees) s += 15;
    if ((_hauteurCm ?? 0) >= 41 && (_hauteurCm ?? 0) <= 50) s += 20;
    if ((_poidsKg ?? 0) >= 18 && (_poidsKg ?? 0) <= 35) s += 10;
    return s.clamp(0, 100);
  }

  int _scoreMastiffBoerbull() {
    int s = 0;
    if (_poil == Poil.rasCourt) s += 10;
    if (_corpulence == Corpulence.massif) s += 25;
    if (_peau == PeauCoussinets.peauLacheDewlap) s += 20;
    if (_museau == Museau.court) s += 15;
    if ((_hauteurCm ?? 0) >= 55) s += 15;
    if ((_poidsKg ?? 0) >= 40) s += 15;
    return s.clamp(0, 100);
  }

  int _scoreTosa() {
    int s = 0;
    if (_poil == Poil.rasCourt) s += 10;
    if (_corpulence == Corpulence.puissantCompact) s += 15;
    if (_peau == PeauCoussinets.peauLacheDewlap) s += 10;
    if (_oreilles == Oreilles.tombantesTriangulaires) s += 15;
    if ((_hauteurCm ?? 0) >= 60) s += 20;
    if ((_poidsKg ?? 0) >= 35) s += 10;
    if (_couleur == Couleur.fauveRouge) s += 10;
    return s.clamp(0, 100);
  }

  int _scoreRott() {
    int s = 0;
    if (_poil == Poil.rasCourt) s += 10;
    if (_corpulence == Corpulence.puissantCompact) s += 20;
    if (_museau == Museau.moyen && _stop != StopTete.prononce) s += 10;
    if (_couleur == Couleur.noirFeu) s += 30;
    if ((_hauteurCm ?? 0) >= 56 && (_hauteurCm ?? 0) <= 70) s += 15;
    if ((_poidsKg ?? 0) >= 35 && (_poidsKg ?? 0) <= 60) s += 15;
    return s.clamp(0, 100);
  }

  // ---------- BUILD ----------
  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(title: const Text("Assistant – chiens dits dangereux")),
      body: SingleChildScrollView(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.stretch,
          children: [
            // Étape 1 : identification simple
            const Text(
              "Identification rapide",
              style: TextStyle(fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 8),
            DropdownButtonFormField<RacePresumee>(
              value: _race,
              isExpanded: true,
              decoration: _dec("Race présumée"),
              items: const [
                DropdownMenuItem(
                  value: RacePresumee.inconnu,
                  child: Text("Inconnue / croisé"),
                ),
                DropdownMenuItem(
                  value: RacePresumee.amstaff,
                  child: Text(
                    "American Staffordshire terrier (amstaff) – ⚠️ à ne pas confondre avec le Staffordshire Bull Terrier (« staffie »)",
                  ),
                ),
                DropdownMenuItem(
                  value: RacePresumee.rottweiler,
                  child: Text("Rottweiler (rott)"),
                ),
                DropdownMenuItem(value: RacePresumee.tosa, child: Text("Tosa")),
                DropdownMenuItem(
                  value: RacePresumee.mastiffBoerbull,
                  child: Text(
                    "Type Mastiff / Boerboel (« boerbull ») – non LOF",
                  ),
                ),
              ],
              onChanged: (v) => setState(() => _race = v!),
            ),
            const SizedBox(height: 8),
            DropdownButtonFormField<LOFStatus>(
              value: _lof,
              isExpanded: true,
              decoration: _dec("Statut LOF (pédigrée)"),
              items: const [
                DropdownMenuItem(
                  value: LOFStatus.inconnu,
                  child: Text("Je ne sais pas"),
                ),
                DropdownMenuItem(
                  value: LOFStatus.lof,
                  child: Text("Inscrit au LOF (Livre des origines français)"),
                ),
                DropdownMenuItem(
                  value: LOFStatus.nonLof,
                  child: Text("Non LOF / sans pédigrée (dit « type »)"),
                ),
              ],
              onChanged: (v) => setState(() => _lof = v!),
            ),
            const SizedBox(height: 8),
            DropdownButtonFormField<AgeBand>(
              value: _age,
              isExpanded: true,
              decoration: _dec("Âge du chien"),
              items: const [
                DropdownMenuItem(
                  value: AgeBand.mois0a7,
                  child: Text("0–7 mois"),
                ),
                DropdownMenuItem(
                  value: AgeBand.mois8a12,
                  child: Text("8–12 mois"),
                ),
                DropdownMenuItem(
                  value: AgeBand.plusDe12,
                  child: Text("> 12 mois"),
                ),
              ],
              onChanged: (v) => setState(() => _age = v!),
            ),
            const SizedBox(height: 8),
            DropdownButtonFormField<ADejaMordu>(
              value: _mordu,
              isExpanded: true,
              decoration: _dec("Le chien a-t-il déjà mordu ?"),
              items: const [
                DropdownMenuItem(
                  value: ADejaMordu.non,
                  child: Text("Non / inconnu"),
                ),
                DropdownMenuItem(
                  value: ADejaMordu.oui,
                  child: Text("Oui – chien mordeur"),
                ),
              ],
              onChanged: (v) => setState(() => _mordu = v!),
            ),

            const SizedBox(height: 16),
            const Text(
              "Questionnaire morphologique (si besoin)",
              style: TextStyle(fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 8),

            Row(
              children: [
                Expanded(
                  child: TextField(
                    decoration: _dec("Hauteur au garrot (cm)", hint: "ex. 48"),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                    onChanged: (t) => _hauteurCm = _parse(t),
                  ),
                ),
                const SizedBox(width: 8),
                Expanded(
                  child: TextField(
                    decoration: _dec("Poids (kg)", hint: "ex. 28"),
                    keyboardType: const TextInputType.numberWithOptions(
                      decimal: true,
                    ),
                    onChanged: (t) => _poidsKg = _parse(t),
                  ),
                ),
              ],
            ),
            const SizedBox(height: 8),

            Wrap(
              spacing: 8,
              runSpacing: 8,
              children: [
                _chip<Couleur>(
                  label: "Couleur",
                  value: _couleur,
                  options: const {
                    Couleur.noirFeu: "Noir & feu",
                    Couleur.fauveRouge: "Fauve / rouge",
                    Couleur.autre: "Autre",
                  },
                  onChanged: (v) => setState(() => _couleur = v),
                ),
                _chip<Museau>(
                  label: "Museau",
                  value: _museau,
                  options: const {
                    Museau.court: "Court",
                    Museau.moyen: "Moyen",
                    Museau.long: "Long",
                  },
                  onChanged: (v) => setState(() => _museau = v),
                ),
                _chip<StopTete>(
                  label: "Stop (front-chanfrein)",
                  value: _stop,
                  options: const {
                    StopTete.peuMarque: "Peu marqué",
                    StopTete.moyen: "Moyen",
                    StopTete.prononce: "Prononcé",
                  },
                  onChanged: (v) => setState(() => _stop = v),
                ),
                _chip<PeauCoussinets>(
                  label: "Peau / fanons",
                  value: _peau,
                  options: const {
                    PeauCoussinets.peauFerme: "Peau ferme",
                    PeauCoussinets.peauLacheDewlap: "Peau lâche / fanons",
                  },
                  onChanged: (v) => setState(() => _peau = v),
                ),
                _chip<Corpulence>(
                  label: "Corpulence",
                  value: _corpulence,
                  options: const {
                    Corpulence.puissantCompact: "Puissant & compact",
                    Corpulence.massif: "Très massif",
                    Corpulence.plutotSec: "Plutôt sec",
                  },
                  onChanged: (v) => setState(() => _corpulence = v),
                ),
                _chip<Oreilles>(
                  label: "Oreilles",
                  value: _oreilles,
                  options: const {
                    Oreilles.roseSemiDressees: "En rose / semi-dressées",
                    Oreilles.tombantesTriangulaires: "Tombantes triangulaires",
                    Oreilles.autres: "Autres",
                  },
                  onChanged: (v) => setState(() => _oreilles = v),
                ),
                _chip<Poil>(
                  label: "Poil",
                  value: _poil,
                  options: const {
                    Poil.rasCourt: "Ras / court",
                    Poil.autre: "Autre",
                  },
                  onChanged: (v) => setState(() => _poil = v),
                ),
              ],
            ),

            const SizedBox(height: 16),
            ElevatedButton.icon(
              onPressed: _qualifier,
              icon: const Icon(Icons.search),
              label: const Text("Identifier & lister les obligations"),
            ),

            if (_result != null) ...[
              const SizedBox(height: 20),
              _badge(),
              const SizedBox(height: 8),
              Text(
                _explication,
                style: const TextStyle(
                  fontStyle: FontStyle.italic,
                  height: 1.2,
                ),
              ),
              const SizedBox(height: 16),

              // Sections + rendu gras markdown
              _section(
                "Obligations du détenteur / propriétaire",
                _obligDetenteur,
              ),
              _section(
                "Obligations de la personne qui promène",
                _obligPromeneur,
              ),
              _section("Obligations pour le chien", _obligChien),
              _section("Pouvoirs / obligations du maire", _pouvoirsMaire),
              if (_notes.isNotEmpty) _section("Notes importantes", _notes),

              const SizedBox(height: 4),
              _defsAcronymes(),
              const SizedBox(height: 8),
              _sourcesOfficielles(),

              const SizedBox(height: 12),
              _disclaimer(),
            ],
          ],
        ),
      ),
    );
  }

  // ---------- Widgets réutilisables ----------
  Widget _badge() {
    Color c;
    String t;
    switch (_result!) {
      case CatResult.cat1:
        c = Colors.red.shade400;
        t = "Proposition : 1ʳᵉ catégorie (chiens d’attaque)";
        break;
      case CatResult.cat2:
        c = Colors.orange.shade400;
        t = "Proposition : 2ᵉ catégorie (garde/défense)";
        break;
      case CatResult.nonCat:
        c = Colors.green.shade500;
        t = "Proposition : non catégorisé";
        break;
      case CatResult.indetermine:
        c = Colors.grey;
        t = "Proposition : indéterminée";
        break;
    }
    return Container(
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: c.withValues(alpha: .15),
        borderRadius: BorderRadius.circular(10),
      ),
      child: Row(
        children: [
          Icon(Icons.pets, color: c),
          const SizedBox(width: 8),
          Expanded(
            child: Text(
              t,
              style: TextStyle(color: c, fontWeight: FontWeight.bold),
            ),
          ),
        ],
      ),
    );
  }

  Widget _section(String title, List<String> items) {
    if (items.isEmpty) return const SizedBox.shrink();
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          title,
          style: const TextStyle(fontWeight: FontWeight.bold, fontSize: 16),
        ),
        const SizedBox(height: 8),
        ...items.map(
          (e) => Padding(
            padding: const EdgeInsets.only(bottom: 6),
            child: Row(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [const Text("• "), Expanded(child: _md(e))],
            ),
          ),
        ),
        const SizedBox(height: 12),
      ],
    );
  }

  Widget _defsAcronymes() {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(12),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            const Text(
              "Définitions & acronymes utiles",
              style: TextStyle(fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 8),
            _md(
              "**LOF** : Livre des origines français (pédigrée). Un chien **LOF** est « de race » ; sans LOF on parle de « **type** ».",
            ),
            const SizedBox(height: 6),
            _md(
              "**I-CAD** : Fichier national d’**identification** des carnivores domestiques (puce/tatouage).",
            ),
            const SizedBox(height: 6),
            _md(
              "**Permis de détention** : délivré par le **maire** du lieu de résidence du détenteur (provisoire entre **8–12 mois**, puis **définitif**).",
            ),
            const SizedBox(height: 6),
            _md(
              "**Formation de 7 h** : attestation d’aptitude pour le détenteur (sécurité, législation, prévention).",
            ),
            const SizedBox(height: 6),
            _md(
              "**Évaluation comportementale** : réalisée par un **vétérinaire** (classement **niveaux 1 à 4**) ; oriente la décision du maire.",
            ),
            const SizedBox(height: 6),
            _md(
              "**Catégorie 1** : chiens **d’attaque** (types non LOF : **amstaff type/pit-bull**, **mastiff/boerboel dit boerbull**, **tosa type**).",
            ),
            const SizedBox(height: 6),
            _md(
              "**Catégorie 2** : chiens de **garde/défense** (**amstaff LOF**, **tosa LOF**, **rottweiler LOF ou assimilable**).",
            ),
          ],
        ),
      ),
    );
  }

  Widget _sourcesOfficielles() {
    return Card(
      child: Padding(
        padding: const EdgeInsets.all(12),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            const Text(
              "Sources officielles",
              style: TextStyle(fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 8),
            TextButton(
              onPressed:
                  () => _open(
                    "https://www.service-public.fr/particuliers/vosdroits/F1839",
                  ),
              child: const Align(
                alignment: Alignment.centerLeft,
                child: Text("Service-public.fr – Chiens dangereux (F1839)"),
              ),
            ),
            TextButton(
              onPressed:
                  () => _open(
                    "https://www.nord.gouv.fr/Demarches/Autres-demarches/Chiens-dangereux/Chiens-dangereux-la-reglementation",
                  ),
              child: const Align(
                alignment: Alignment.centerLeft,
                child: Text(
                  "Préfecture du Nord – Chiens dangereux : la réglementation",
                ),
              ),
            ),
            TextButton(
              onPressed:
                  () => _open(
                    "https://www.legifrance.gouv.fr/loda/id/JORFTEXT000000210847#LEGIARTI000006608538",
                  ),
              child: const Align(
                alignment: Alignment.centerLeft,
                child: Text(
                  "Légifrance – dispositions (Code rural / art. L.211-11 s.)",
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _disclaimer() {
    return Container(
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: Theme.of(context).colorScheme.surfaceVariant,
        borderRadius: BorderRadius.circular(10),
      ),
      child: _md(
        "⚠️ **Outil d’orientation** : la qualification officielle peut nécessiter l’examen des pièces (**LOF**, pédigrée), l’avis du vétérinaire/éleveur et, le cas échéant, l’**évaluation comportementale**. "
        "Toujours vérifier les **arrêtés locaux** et la **décision du maire**.",
      ),
    );
  }

  Widget _chip<T>({
    required String label,
    required T value,
    required Map<T, String> options,
    required void Function(T) onChanged,
  }) {
    return Wrap(
      crossAxisAlignment: WrapCrossAlignment.center,
      children: [
        Padding(
          padding: const EdgeInsets.only(right: 6, bottom: 6),
          child: Text("$label :"),
        ),
        Wrap(
          spacing: 6,
          children:
              options.entries.map((e) {
                final selected = e.key == value;
                return ChoiceChip(
                  label: Text(e.value),
                  selected: selected,
                  onSelected: (_) => onChanged(e.key),
                );
              }).toList(),
        ),
      ],
    );
  }
}
