import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';

/// Result of parsing a NATINF payload coming from the server.
class ParsedServerNatinf {
  const ParsedServerNatinf({
    required this.natinf,
    required this.numero,
    required this.docsProAvailable,
    required this.docsProUnauthorized,
    required this.references,
  });

  final Natinf natinf;
  final String? numero;
  final bool docsProAvailable;
  final bool docsProUnauthorized;
  final List<NatinfReference> references;
}

/// Utility that ensures a list of dynamic values is converted into concrete
/// [Natinf] instances. Raw maps coming from persistence layers are converted
/// using `Natinf.fromJson`, while existing instances are preserved as-is.
List<Natinf> materializeNatinfs(List<dynamic>? rawList) {
  if (rawList == null || rawList.isEmpty) {
    return const [];
  }

  return rawList
      .map(
        (value) =>
            value is Natinf
                ? value
                : parseStoredNatinf(Map<String, dynamic>.from(value as Map)),
      )
      .toList(growable: false);
}

/// Normalizes a server payload into a [_ParsedServerNatinf] tuple, ensuring the
/// resulting [Natinf] object no longer contains ephemeral API-only fields.
ParsedServerNatinf parseServerNatinf(Map<String, dynamic> raw) {
  final working = Map<String, dynamic>.from(raw);
  final numero = working['numero_natinf'] as String?;
  final docsProAvailableRaw = working['docs_pro_available'];
  final docsProAvailable =
      docsProAvailableRaw == true || docsProAvailableRaw == 'true';
  final docsPro = working['docs_pro'];
  final docsProUnauthorized =
      docsProAvailable &&
      docsPro is Map &&
      (docsPro['error'] == 'Unauthorized' ||
          docsPro['error'] == 'unauthorized');

  final referencesRaw = working['references'];
  final references =
      referencesRaw is List
          ? referencesRaw
              .whereType<Map>()
              .map((ref) => Map<String, dynamic>.from(ref))
              .where((ref) => ref['url'] != null)
              .map((ref) => NatinfReference.fromJson(ref))
              .toList(growable: false)
          : const <NatinfReference>[];

  working
    ..remove('docs_pro_available')
    ..remove('docs_pro')
    ..remove('categories')
    ..remove('references');

  return ParsedServerNatinf(
    natinf: Natinf.fromJson(normalizeNatinfJson(working)),
    numero: numero,
    docsProAvailable: docsProAvailable,
    docsProUnauthorized: docsProUnauthorized,
    references: references,
  );
}

/// Converts a locally stored NATINF payload into a [Natinf], adding missing
/// required fields coming from legacy schemas (e.g. `dacg_source`).
Natinf parseStoredNatinf(Map<String, dynamic> raw) {
  return Natinf.fromJson(normalizeNatinfJson(raw));
}

/// Ensures a NATINF JSON payload contains the mandatory fields expected by the
/// generated model while sanitizing nested collections.
Map<String, dynamic> normalizeNatinfJson(Map<String, dynamic> json) {
  final normalized = Map<String, dynamic>.from(json);
  normalized.putIfAbsent('dacg_source', () => 0);

  final references = normalized['references'];
  if (references is List) {
    final sanitized = references
        .whereType<Map>()
        .where((ref) => ref['url'] != null)
        .map((ref) => Map<String, dynamic>.from(ref))
        .toList(growable: false);
    normalized['references'] = sanitized;
  } else {
    normalized.remove('references');
  }

  return normalized;
}
