import 'dart:async';
import 'dart:io';
import 'package:diacritic/diacritic.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_mappers.dart';
import 'package:natinfo_flutter/features/natinf/domain/entities/favourite_folder.dart';
import 'package:meta/meta.dart';
import 'package:path/path.dart';
import 'package:path_provider/path_provider.dart';
import 'package:sembast/sembast.dart';
import 'package:sembast/sembast_io.dart';
import 'package:uuid/uuid.dart';

class _Scored {
  final Natinf natinf;
  final int relevance;
  final int qualificationPos;
  final int pk;

  _Scored({
    required this.natinf,
    required this.relevance,
    required this.qualificationPos,
    required this.pk,
  });
}

class DatabaseHelper {
  static final DatabaseHelper _singleton = DatabaseHelper._internal();
  factory DatabaseHelper() => _singleton;
  DatabaseHelper._internal();

  Database? _db;
  final _store = intMapStoreFactory.store('natinf');
  final _historyStore = intMapStoreFactory.store('historique');
  final _favouritesStore = intMapStoreFactory.store('favourites');
  final _favouriteFoldersStore = intMapStoreFactory.store('favourite_folders');
  final _favouriteFolderItemsStore = stringMapStoreFactory.store(
    'favourite_folder_items',
  );
  final _docsProStore = stringMapStoreFactory.store('docs_pro');
  final _referencesStore = stringMapStoreFactory.store('references');
  final _categoriesStore = intMapStoreFactory.store('categories');
  final _favoritesStateStore = stringMapStoreFactory.store('favorites_state');

  final _uuid = const Uuid();

  /// Initialise la base de données Sembast
  Future<void> initDb() async {
    await _db?.close();
    final Directory dir = await getApplicationDocumentsDirectory();
    final dbPath = join(dir.path, 'natinfo.db');
    _db = await databaseFactoryIo.openDatabase(dbPath);
    await _runMigrations();
  }

  /// Initialise une base en mémoire pour les tests.
  @visibleForTesting
  Future<void> initDbForTesting(
    DatabaseFactory databaseFactory,
    String path,
  ) async {
    await _db?.close();
    _db = await databaseFactory.openDatabase(path);
    await _runMigrations();
  }

  /// Expose la base de données pour les assertions de tests.
  @visibleForTesting
  Database get dbForTesting => _dbOrThrow;

  Database get _dbOrThrow {
    final db = _db;
    if (db == null) {
      throw Exception("La base de données n'est pas initialisée.");
    }
    return db;
  }

  Future<void> _runMigrations() async {
    await _migrateFavoritesSchema();
  }

  Future<int> insertOrUpdateNatinf(Natinf natinf) async {
    final record = _store.record(natinf.id!);
    await record.put(_dbOrThrow, natinf.toJson());
    return natinf.id!;
  }

  Future<void> setDocsProInfo(
    String numero, {
    required bool unauthorized,
    required bool available,
  }) async {
    final db = _dbOrThrow;
    await _docsProStore.record(numero).put(db, {
      'unauthorized': unauthorized,
      'available': available,
    });
  }

  Future<Map<String, dynamic>?> getDocsProInfo(String numero) async {
    final db = _dbOrThrow;
    return await _docsProStore.record(numero).get(db) as Map<String, dynamic>?;
  }

  Future<Map<String, bool>> loadDocsProUnauthorizedMap() async {
    final db = _dbOrThrow;
    final records = await _docsProStore.find(db);
    final result = <String, bool>{};
    for (final rec in records) {
      final map = rec.value as Map<String, dynamic>;
      final unauth = map['unauthorized'] == true;
      result[rec.key as String] = unauth;
    }
    return result;
  }

  Future<Map<String, bool>> loadDocsProAvailableMap() async {
    final db = _dbOrThrow;
    final records = await _docsProStore.find(db);
    final result = <String, bool>{};
    for (final rec in records) {
      final map = rec.value as Map<String, dynamic>;
      final available = map['available'] == true;
      result[rec.key as String] = available;
    }
    return result;
  }

  Future<void> clearDocsPro() async {
    final db = _dbOrThrow;
    await _docsProStore.delete(db);
  }

  Future<void> setReferences(String numero, List<NatinfReference> refs) async {
    final db = _dbOrThrow;
    await _referencesStore.record(numero).put(db, {
      'list': refs.map((e) => e.toJson()).toList(),
    });
  }

  Future<List<NatinfReference>?> getReferences(String numero) async {
    final db = _dbOrThrow;
    final Map<String, dynamic>? map =
        await _referencesStore.record(numero).get(db) as Map<String, dynamic>?;
    if (map != null) {
      final data = map['list'];
      if (data is List) {
        final list =
            data
                .map(
                  (e) => NatinfReference.fromJson(
                    Map<String, dynamic>.from(e as Map),
                  ),
                )
                .toList();
        return list;
      }
    }
    return null;
  }

  Future<Map<String, List<NatinfReference>>> loadReferencesMap() async {
    final db = _dbOrThrow;
    final records = await _referencesStore.find(db);
    final result = <String, List<NatinfReference>>{};
    for (final rec in records) {
      final map = rec.value as Map<String, dynamic>?;
      if (map != null) {
        final data = map['list'];
        final list =
            data is List
                ? data
                    .map(
                      (e) => NatinfReference.fromJson(
                        Map<String, dynamic>.from(e as Map),
                      ),
                    )
                    .toList()
                : <NatinfReference>[];
        result[rec.key as String] = list;
      }
    }
    return result;
  }

  Future<int> insertOrUpdateCategory(Category category) async {
    final record = _categoriesStore.record(category.id!);
    await record.put(_dbOrThrow, category.toJson());
    return category.id!;
  }

  Future<Category?> getCategory(int id) async {
    final db = _dbOrThrow;
    final Map<String, dynamic>? map =
        await _categoriesStore.record(id).get(db) as Map<String, dynamic>?;
    if (map != null) {
      return Category.fromJson(Map<String, dynamic>.from(map));
    }
    return null;
  }

  Future<List<CategoryBrief>> getRootCategories() async {
    final db = _dbOrThrow;
    final finder = Finder(
      filter: Filter.equals('parent', null),
      sortOrders: [SortOrder('name')],
    );
    final records = await _categoriesStore.find(db, finder: finder);
    return records
        .map(
          (rec) => CategoryBrief.fromJson(
            Map<String, dynamic>.from(rec.value as Map),
          ),
        )
        .toList();
  }

  Future<void> clearCategories() async {
    final db = _dbOrThrow;
    await _categoriesStore.delete(db);
  }

  Future<int> countCategories() async {
    final db = _dbOrThrow;
    return _categoriesStore.count(db);
  }

  Future<Natinf?> getNatinfByNumero(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    final record = await _store.findFirst(db, finder: finder);
    if (record != null) {
      final map = Map<String, dynamic>.from(record.value as Map);
      var natinf = parseStoredNatinf(map);
      final refs = await getReferences(numero);
      if (refs != null && refs.isNotEmpty) {
        natinf = natinf.copyWith(references: refs);
      }
      return natinf;
    }
    return null;
  }

  Future<int> countItems() async {
    final db = _dbOrThrow;
    final count = await _store.count(db);
    return count;
  }

  Future<void> clearOfflineData() async {
    final db = _dbOrThrow;
    // Supprime les NATINF téléchargés
    await _store.delete(db);
    // Supprime également le cache des catégories associé
    await _categoriesStore.delete(db);
    print("Données locales supprimées (NATINF + catégories).");
  }

  Future<void> clearNatinfData() async {
    final db = _dbOrThrow;
    await _store.delete(db);
  }

  Future<bool> isEmpty() async {
    final db = _dbOrThrow;
    final count = await _store.count(db);
    return count == 0;
  }

  /// Recherche locale en essayant de reproduire la logique de l'API NATINFo
  Future<List<Natinf>> getNatinfByQuery(String query) async {
    final db = _dbOrThrow;
    String normalize(String s) =>
        removeDiacritics(
          s,
        ).toLowerCase().replaceAll(RegExp(r'[^\w]+'), ' ').trim();

    final normalizedQuery = normalize(query);
    final queryWords =
        normalizedQuery.split(' ').where((w) => w.isNotEmpty).toList();

    RegExp makePattern(String kw) {
      var p = RegExp.escape(kw);
      p = p.replaceAllMapped(
        RegExp(r'([A-Za-z])\\?\.?([0-9])'),
        (m) => '${m[1]}\\.?${m[2]}',
      );
      return RegExp(r'\b' + p, caseSensitive: false);
    }

    final List<RegExp> patterns = [];
    for (final word in queryWords) {
      patterns.add(makePattern(word));
    }

    final fullQueryRegex = RegExp(
      r'\b' + RegExp.escape(normalizedQuery) + r'\b',
      caseSensitive: false,
    );

    final records = await _store.find(db);
    final buffer = <_Scored>[];

    for (final record in records) {
      final map = Map<String, dynamic>.from(record.value as Map);
      final normalized = normalizeNatinfJson(map);

      final numero = normalize(normalized['numero_natinf']?.toString() ?? '');
      final nature = normalize(
        normalized['nature_infraction']?.toString() ?? '',
      );
      final qualification = normalize(
        normalized['qualification_infraction']?.toString() ?? '',
      );
      final definiePar = normalize(normalized['definie_par']?.toString() ?? '');
      final reprimeePar = normalize(
        normalized['reprimee_par']?.toString() ?? '',
      );

      final bool allMatch = patterns.every(
        (pat) =>
            pat.hasMatch(numero) ||
            pat.hasMatch(nature) ||
            pat.hasMatch(qualification) ||
            pat.hasMatch(definiePar) ||
            pat.hasMatch(reprimeePar),
      );

      if (!allMatch) continue;

      int relevance = 0;
      if (fullQueryRegex.hasMatch(numero)) {
        relevance = 100;
      } else if (fullQueryRegex.hasMatch(qualification)) {
        relevance = 90;
      } else if (fullQueryRegex.hasMatch(definiePar)) {
        relevance = 80;
      } else if (fullQueryRegex.hasMatch(reprimeePar)) {
        relevance = 70;
      }

      int qualificationPos = qualification.indexOf(normalizedQuery);
      if (qualificationPos == -1) qualificationPos = 100000;

      buffer.add(
        _Scored(
          natinf: parseStoredNatinf(normalized),
          relevance: relevance,
          qualificationPos: qualificationPos,
          pk: record.key,
        ),
      );
    }

    buffer.sort((a, b) {
      final rel = b.relevance.compareTo(a.relevance);
      if (rel != 0) return rel;

      final pos = a.qualificationPos.compareTo(b.qualificationPos);
      if (pos != 0) return pos;

      return a.pk.compareTo(b.pk);
    });

    return buffer.map((s) => s.natinf).toList();
  }

  /// Favoris
  Future<void> _migrateFavoritesSchema() async {
    final db = _dbOrThrow;
    final schemaRecord = _favoritesStateStore.record('schema');
    final rawSchema = await schemaRecord.get(db) as Map<String, dynamic>?;
    final currentVersion = rawSchema?['favorites_version'] as int? ?? 0;
    if (currentVersion >= 2) {
      return;
    }

    final nowMs = DateTime.now().millisecondsSinceEpoch;

    // Folders: assign uuid, updated_at, position, deleted=false
    final folderRecords = await _favouriteFoldersStore.find(db);
    final folderUuidById = <int, String>{};
    final foldersByParent =
        <int?, List<RecordSnapshot<int, Map<String, dynamic>>>>{};
    for (final rec in folderRecords) {
      final parentId = (rec.value['parent_id'] as int?);
      final bucket = foldersByParent.putIfAbsent(
        parentId,
        () => <RecordSnapshot<int, Map<String, dynamic>>>[],
      );
      bucket.add(rec);
    }
    for (final entry in foldersByParent.entries) {
      entry.value.sort((a, b) {
        final aCreated = a.value['created_at'] as int? ?? 0;
        final bCreated = b.value['created_at'] as int? ?? 0;
        return aCreated.compareTo(bCreated);
      });
      for (var i = 0; i < entry.value.length; i++) {
        final rec = entry.value[i];
        final map = Map<String, dynamic>.from(rec.value as Map);
        final uuid = (map['uuid'] as String?)?.trim();
        final assigned = (uuid != null && uuid.isNotEmpty) ? uuid : _uuid.v4();
        final updated = map['updated_at'] ?? map['created_at'] ?? nowMs;
        map['uuid'] = assigned;
        map['updated_at'] = updated;
        map['position'] = map['position'] is int ? map['position'] : i;
        map['deleted'] = map['deleted'] == true ? true : false;
        await _favouriteFoldersStore.record(rec.key).put(db, map);
        folderUuidById[rec.key as int] = assigned;
      }
    }

    // Root favourites: assign uuid/id, updated_at, position, deleted=false
    final favRecords = await _favouritesStore.find(db);
    favRecords.sort((a, b) {
      final aAdded = a.value['added_at'] as int? ?? 0;
      final bAdded = b.value['added_at'] as int? ?? 0;
      return aAdded.compareTo(bAdded);
    });
    for (var i = 0; i < favRecords.length; i++) {
      final rec = favRecords[i];
      final map = Map<String, dynamic>.from(rec.value as Map);
      final rawId = (map['uuid'] as String?)?.trim() ?? (map['id'] as String?);
      final assignedId =
          (rawId != null && rawId.isNotEmpty) ? rawId : _uuid.v4();
      final updated =
          map['updated_at'] ?? map['updatedAt'] ?? map['added_at'] ?? nowMs;
      map['uuid'] = assignedId;
      map['id'] = assignedId;
      map['updated_at'] = updated;
      map['position'] = map['position'] is int ? map['position'] : i;
      map['deleted'] = map['deleted'] == true ? true : false;
      await _favouritesStore.record(rec.key).put(db, map);
    }

    // Folder items: assign uuid/id, updated_at, position, folder_uuid
    final folderItems = await _favouriteFolderItemsStore.find(db);
    final itemsByFolder =
        <int?, List<RecordSnapshot<String, Map<String, dynamic>>>>{};
    for (final rec in folderItems) {
      final map = rec.value as Map<String, dynamic>;
      final folderId = map['folder_id'] as int?;
      final bucket = itemsByFolder.putIfAbsent(
        folderId,
        () => <RecordSnapshot<String, Map<String, dynamic>>>[],
      );
      bucket.add(rec);
    }
    for (final entry in itemsByFolder.entries) {
      entry.value.sort((a, b) {
        final aAdded = a.value['added_at'] as int? ?? 0;
        final bAdded = b.value['added_at'] as int? ?? 0;
        return aAdded.compareTo(bAdded);
      });
      for (var i = 0; i < entry.value.length; i++) {
        final rec = entry.value[i];
        final map = Map<String, dynamic>.from(rec.value as Map);
        final rawId =
            (map['uuid'] as String?)?.trim() ?? (map['id'] as String?);
        final assignedId =
            (rawId != null && rawId.isNotEmpty) ? rawId : _uuid.v4();
        final updated =
            map['updated_at'] ?? map['updatedAt'] ?? map['added_at'] ?? nowMs;
        final folderId = map['folder_id'] as int?;
        map['uuid'] = assignedId;
        map['id'] = assignedId;
        map['updated_at'] = updated;
        map['position'] = map['position'] is int ? map['position'] : i;
        map['deleted'] = map['deleted'] == true ? true : false;
        final folderUuid =
            map['folder_uuid'] as String? ??
            (folderId != null ? folderUuidById[folderId] : null);
        if (folderUuid != null) {
          map['folder_uuid'] = folderUuid;
        }
        await _favouriteFolderItemsStore.record(rec.key).put(db, map);
      }
    }

    final newSchema = Map<String, dynamic>.from(rawSchema ?? {});
    newSchema['favorites_version'] = 2;
    await schemaRecord.put(db, newSchema);
  }

  Future<int> addFavourites(String numero) async {
    final db = _dbOrThrow;
    final count = await _favouritesStore.count(db);
    final nowMs = DateTime.now().millisecondsSinceEpoch;
    final id = _uuid.v4();
    return await _favouritesStore.add(db, {
      'numero_natinf': numero,
      'added_at': nowMs,
      'updated_at': nowMs,
      'uuid': id,
      'id': id,
      'position': count,
      'deleted': false,
    });
  }

  Future<List<Map<String, dynamic>>> getFavourites() async {
    final db = _dbOrThrow;
    final finder = Finder(sortOrders: [SortOrder('added_at', false)]);
    final records = await _favouritesStore.find(db, finder: finder);
    return records
        .map((record) => record.value as Map<String, dynamic>)
        .where((value) => value['deleted'] != true)
        .toList();
  }

  Future<bool> isFavourite(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    final record = await _favouritesStore.findFirst(db, finder: finder);
    if (record == null) return false;
    final map = record.value as Map<String, dynamic>;
    return map['deleted'] != true;
  }

  Future<void> removeFavourite(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    await _favouritesStore.delete(db, finder: finder);
    await removeNatinfFromAllFolders(numero);
  }

  Future<void> clearFavourites({bool includeFolders = false}) async {
    final db = _dbOrThrow;
    await _favouritesStore.delete(db);
    await _favouriteFolderItemsStore.delete(db);
    if (includeFolders) {
      await _favouriteFoldersStore.delete(db);
    }
  }

  /// Crée un dossier de favoris et renvoie son identifiant.
  Future<int> createFavouriteFolder({
    required String name,
    int? parentId,
  }) async {
    final db = _dbOrThrow;
    final parentFilter =
        parentId == null
            ? Filter.equals('parent_id', null)
            : Filter.equals('parent_id', parentId);
    final siblings = await _favouriteFoldersStore.find(
      db,
      finder: Finder(filter: parentFilter),
    );
    final position = siblings.length;
    final nowMs = DateTime.now().millisecondsSinceEpoch;
    final uuid = _uuid.v4();
    return await _favouriteFoldersStore.add(db, {
      'name': name,
      'parent_id': parentId,
      'created_at': nowMs,
      'updated_at': nowMs,
      'uuid': uuid,
      'position': position,
      'deleted': false,
    });
  }

  /// Renvoie l'ensemble des dossiers de favoris.
  Future<List<FavouriteFolder>> getAllFavouriteFolders() async {
    final db = _dbOrThrow;
    final records = await _favouriteFoldersStore.find(db);
    return records
        .map(
          (record) => FavouriteFolder.fromMap(
            id: record.key as int,
            map: Map<String, dynamic>.from(record.value as Map),
          ),
        )
        .where((folder) => folder.deleted == false)
        .toList();
  }

  /// Modifie le nom d'un dossier de favoris.
  Future<void> renameFavouriteFolder(int folderId, String newName) async {
    final db = _dbOrThrow;
    await _favouriteFoldersStore.record(folderId).update(db, {
      'name': newName,
      'updated_at': DateTime.now().millisecondsSinceEpoch,
    });
  }

  /// Supprime un dossier ainsi que son sous-arborescence.
  Future<void> deleteFavouriteFolder(int folderId) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('parent_id', folderId));
    final children = await _favouriteFoldersStore.find(db, finder: finder);
    for (final child in children) {
      await deleteFavouriteFolder(child.key as int);
    }
    final nowMs = DateTime.now().millisecondsSinceEpoch;
    await _markFolderDeleted(folderId, nowMs);
  }

  /// Ajoute un NATINF dans un dossier donné.
  Future<void> addNatinfToFolder(int folderId, String numero) async {
    final db = _dbOrThrow;
    final key = '$folderId|$numero';
    final finder = Finder(filter: Filter.equals('folder_id', folderId));
    final existing = await _favouriteFolderItemsStore.find(db, finder: finder);
    final position = existing.length;
    final nowMs = DateTime.now().millisecondsSinceEpoch;
    final id = _uuid.v4();
    await _favouriteFolderItemsStore.record(key).put(db, {
      'folder_id': folderId,
      'numero_natinf': numero,
      'added_at': nowMs,
      'updated_at': nowMs,
      'uuid': id,
      'id': id,
      'position': position,
      'deleted': false,
      'folder_uuid': await _folderUuid(folderId),
    });
  }

  /// Retire un NATINF d'un dossier.
  Future<void> removeNatinfFromFolder(int folderId, String numero) async {
    final db = _dbOrThrow;
    final key = '$folderId|$numero';
    await _favouriteFolderItemsStore.record(key).delete(db);
  }

  /// Retire un NATINF de tous les dossiers.
  Future<void> removeNatinfFromAllFolders(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    await _favouriteFolderItemsStore.delete(db, finder: finder);
  }

  /// Renvoie la liste des identifiants de dossiers contenant [numero].
  Future<Set<int>> getFolderIdsForNatinf(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    final records = await _favouriteFolderItemsStore.find(db, finder: finder);
    return records
        .map((record) {
          final map = record.value as Map<String, dynamic>;
          if (map['deleted'] == true) {
            return null;
          }
          return map['folder_id'] as int;
        })
        .whereType<int>()
        .toSet();
  }

  /// Renvoie les NATINF présents dans chaque dossier.
  Future<Map<int, List<String>>> getFolderNatinfMap() async {
    final db = _dbOrThrow;
    final records = await _favouriteFolderItemsStore.find(db);
    final result = <int, List<String>>{};
    for (final record in records) {
      final map = record.value as Map<String, dynamic>;
      if (map['deleted'] == true) continue;
      final folderId = map['folder_id'] as int;
      final numero = map['numero_natinf'] as String?;
      if (numero == null) continue;
      final list = result.putIfAbsent(folderId, () => <String>[]);
      if (!list.contains(numero)) {
        list.add(numero);
      }
    }
    for (final entry in result.entries) {
      entry.value.sort();
    }
    return result;
  }

  /// Renvoie les dossiers bruts (y compris tombstones si demandé).
  Future<List<Map<String, dynamic>>> getFavouriteFoldersRaw({
    bool includeDeleted = true,
  }) async {
    final db = _dbOrThrow;
    final records = await _favouriteFoldersStore.find(db);
    return records
        .map((rec) {
          final map = Map<String, dynamic>.from(rec.value as Map);
          map['id'] = rec.key;
          return map;
        })
        .where((m) => includeDeleted || m['deleted'] != true)
        .toList();
  }

  /// Renvoie les favoris racine bruts (y compris tombstones si demandé).
  Future<List<Map<String, dynamic>>> getFavouriteRootRaw({
    bool includeDeleted = true,
  }) async {
    final db = _dbOrThrow;
    final records = await _favouritesStore.find(db);
    return records
        .map((rec) {
          final map = Map<String, dynamic>.from(rec.value as Map);
          map['id'] = rec.key;
          return map;
        })
        .where((m) => includeDeleted || m['deleted'] != true)
        .toList();
  }

  /// Renvoie les liaisons dossier->NATINF brutes (y compris tombstones si demandé).
  Future<List<Map<String, dynamic>>> getFavouriteFolderItemsRaw({
    bool includeDeleted = true,
  }) async {
    final db = _dbOrThrow;
    final records = await _favouriteFolderItemsStore.find(db);
    return records
        .map((rec) => Map<String, dynamic>.from(rec.value as Map))
        .where((m) => includeDeleted || m['deleted'] != true)
        .toList();
  }

  /// Insère un dossier de favoris provenant d'un snapshot serveur.
  Future<int> insertFavouriteFolderSnapshot({
    required String uuid,
    required String name,
    int? parentId,
    required int position,
    required int updatedAt,
    required bool deleted,
  }) async {
    final db = _dbOrThrow;
    return _favouriteFoldersStore.add(db, {
      'name': name,
      'parent_id': parentId,
      'created_at': updatedAt,
      'updated_at': updatedAt,
      'uuid': uuid,
      'position': position,
      'deleted': deleted,
    });
  }

  /// Insère un favori racine à partir d'un snapshot serveur.
  Future<void> insertRootFavouriteSnapshot({
    required String uuid,
    required String numero,
    required int position,
    required int updatedAt,
    required bool deleted,
  }) async {
    final db = _dbOrThrow;
    await _favouritesStore.add(db, {
      'uuid': uuid,
      'id': uuid,
      'numero_natinf': numero,
      'added_at': updatedAt,
      'updated_at': updatedAt,
      'position': position,
      'deleted': deleted,
    });
  }

  /// Insère une liaison dossier/NATINF à partir d'un snapshot serveur.
  Future<void> insertFolderFavouriteSnapshot({
    required int folderId,
    required String folderUuid,
    required String uuid,
    required String numero,
    required int position,
    required int updatedAt,
    required bool deleted,
  }) async {
    final db = _dbOrThrow;
    final key = '$folderId|$numero';
    await _favouriteFolderItemsStore.record(key).put(db, {
      'folder_id': folderId,
      'numero_natinf': numero,
      'added_at': updatedAt,
      'updated_at': updatedAt,
      'uuid': uuid,
      'id': uuid,
      'position': position,
      'deleted': deleted,
      'folder_uuid': folderUuid,
    });
  }

  /// Récupère l'état d'UI associé aux favoris (tri, filtres...).
  Future<Map<String, dynamic>> getFavoritesState() async {
    final db = _dbOrThrow;
    final Map<String, dynamic>? map =
        await _favoritesStateStore.record('state').get(db)
            as Map<String, dynamic>?;
    return Map<String, dynamic>.from(map ?? {});
  }

  /// Sauvegarde l'état d'UI associé aux favoris.
  Future<void> saveFavoritesState(Map<String, dynamic> state) async {
    final db = _dbOrThrow;
    await _favoritesStateStore.record('state').put(db, state);
  }

  /// Remplace complètement les données de favoris (dossiers, favoris racine, items).
  Future<void> replaceFavoritesData({
    required List<Map<String, dynamic>> folders,
    required List<Map<String, dynamic>> rootItems,
    required List<Map<String, dynamic>> folderItems,
  }) async {
    final db = _dbOrThrow;
    await clearFavourites(includeFolders: true);

    final idByUuid = <String, int>{};
    final sortedFolders = List<Map<String, dynamic>>.from(folders)..sort(
      (a, b) => (a['parent_uuid'] ?? '').toString().compareTo(
        (b['parent_uuid'] ?? '').toString(),
      ),
    );

    for (final folder in sortedFolders) {
      final parentUuid = folder['parent_uuid'] as String?;
      final parentId = parentUuid != null ? idByUuid[parentUuid] : null;
      final nowMs = DateTime.now().millisecondsSinceEpoch;
      final id = await _favouriteFoldersStore.add(db, {
        'uuid': folder['uuid'],
        'name': folder['name'] ?? '',
        'parent_id': parentId,
        'created_at': folder['created_at'] ?? nowMs,
        'updated_at': folder['updated_at'] ?? nowMs,
        'position': folder['position'] ?? 0,
        'deleted': folder['deleted'] == true,
      });
      final folderUuid = folder['uuid'];
      if (folderUuid is String && folderUuid.isNotEmpty) {
        idByUuid[folderUuid] = id;
      }
    }

    for (final fav in rootItems) {
      final nowMs = DateTime.now().millisecondsSinceEpoch;
      await _favouritesStore.add(db, {
        'uuid': fav['uuid'],
        'id': fav['uuid'],
        'numero_natinf': fav['numero'],
        'position': fav['position'] ?? 0,
        'added_at': fav['created_at'] ?? nowMs,
        'updated_at': fav['updated_at'] ?? nowMs,
        'deleted': fav['deleted'] == true,
      });
    }

    for (final item in folderItems) {
      final folderUuid = item['folder_uuid'] as String?;
      final folderId = folderUuid != null ? idByUuid[folderUuid] : null;
      if (folderId == null) continue;
      final numero = item['numero'] as String? ?? '';
      if (numero.isEmpty) continue;
      final nowMs = DateTime.now().millisecondsSinceEpoch;
      final key = '$folderId|$numero';
      await _favouriteFolderItemsStore.record(key).put(db, {
        'folder_id': folderId,
        'folder_uuid': folderUuid,
        'numero_natinf': numero,
        'uuid': item['uuid'],
        'id': item['uuid'],
        'position': item['position'] ?? 0,
        'added_at': item['created_at'] ?? nowMs,
        'updated_at': item['updated_at'] ?? nowMs,
        'deleted': item['deleted'] == true,
      });
    }
  }

  Future<String?> _folderUuid(int folderId) async {
    final db = _dbOrThrow;
    final Map<String, dynamic>? map =
        await _favouriteFoldersStore.record(folderId).get(db)
            as Map<String, dynamic>?;
    if (map == null) return null;
    final uuid = map['uuid'] as String?;
    return (uuid != null && uuid.isNotEmpty) ? uuid : null;
  }

  Future<void> _removeFolderContents(int folderId) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('folder_id', folderId));
    final items = await _favouriteFolderItemsStore.find(db, finder: finder);
    for (final item in items) {
      final map = Map<String, dynamic>.from(item.value as Map);
      map['deleted'] = true;
      map['updated_at'] = DateTime.now().millisecondsSinceEpoch;
      await _favouriteFolderItemsStore.record(item.key).put(db, map);
    }
  }

  Future<void> _markFolderDeleted(int folderId, int timestampMs) async {
    final db = _dbOrThrow;
    await _removeFolderContents(folderId);
    final record = await _favouriteFoldersStore.record(folderId).get(db);
    if (record != null) {
      final map = Map<String, dynamic>.from(record as Map);
      map['deleted'] = true;
      map['updated_at'] = timestampMs;
      await _favouriteFoldersStore.record(folderId).put(db, map);
    }
  }

  /// Historique

  Future<int> addHistory(String numero) async {
    final db = _dbOrThrow;
    return await _historyStore.add(db, {
      'numero_natinf': numero,
      'opened_at': DateTime.now().millisecondsSinceEpoch,
    });
  }

  Future<List<Map<String, dynamic>>> getHistory() async {
    final db = _dbOrThrow;
    final finder = Finder(sortOrders: [SortOrder('opened_at', false)]);
    final records = await _historyStore.find(db, finder: finder);
    return records
        .map((record) => record.value as Map<String, dynamic>)
        .toList();
  }

  Future<void> clearHistory() async {
    final db = _dbOrThrow;
    await _historyStore.delete(db);
  }
}
