import 'package:azlistview/azlistview.dart';
import 'package:diacritic/diacritic.dart';
import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/glossary/data/acronym_service.dart';
import 'package:natinfo_flutter/features/glossary/domain/entities/acronym.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:scrollable_positioned_list/scrollable_positioned_list.dart';

/// Displays the glossary of acronyms with search and alphabetical navigation.
class GlossairePage extends StatefulWidget {
  const GlossairePage({super.key, this.service, this.positionsListener});

  /// Custom service mainly used for testing to inject canned data.
  final AcronymService? service;

  /// Overrides the internal list position listener (useful for tests).
  final ItemPositionsListener? positionsListener;

  @override
  State<GlossairePage> createState() => _GlossairePageState();
}

class _GlossairePageState extends State<GlossairePage> {
  final _controller = TextEditingController();
  late final AcronymService _service;
  late final ItemPositionsListener _itemPositionsListener;
  final ItemScrollController _itemScrollController = ItemScrollController();
  List<_GlossaryItem> _items = [];
  List<String> _indexBarData = [];
  bool _loading = true;

  @override
  void initState() {
    super.initState();
    _service = widget.service ?? AcronymService();
    _itemPositionsListener =
        widget.positionsListener ?? ItemPositionsListener.create();
    _search('');
    MatomoService().trackPage(title: 'Glossaire', path: '/tools/glossaire/');
  }

  @override
  void dispose() {
    _controller.dispose();
    super.dispose();
  }

  Future<void> _search(String query) async {
    setState(() => _loading = true);
    final res = await _service.search(query);
    setState(() {
      _loading = false;
      _items = _buildItems(res);
      _indexBarData = SuspensionUtil.getTagIndexList(_items);
    });
  }

  List<_GlossaryItem> _buildItems(List<Acronym> acronyms) {
    final items =
        acronyms
            .map((a) => _GlossaryItem(acronym: a, tag: _letterFor(a)))
            .toList();
    SuspensionUtil.sortListBySuspensionTag(items);
    SuspensionUtil.setShowSuspensionStatus(items);
    return items;
  }

  String _letterFor(Acronym acronym) {
    final graphemes = acronym.acronym.characters;
    if (graphemes.isEmpty) {
      return '#';
    }
    final raw = removeDiacritics(graphemes.first).toUpperCase();
    final letters = raw.characters;
    if (letters.isEmpty) {
      return '#';
    }
    final normalized = letters.first;
    return RegExp(r'^[A-Z]$').hasMatch(normalized) ? normalized : '#';
  }

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    return Scaffold(
      appBar: AppBar(title: const Text('Glossaire')),
      body: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          children: [
            TextField(
              controller: _controller,
              decoration: const InputDecoration(
                labelText: 'Rechercher un acronyme',
                border: OutlineInputBorder(),
              ),
              onChanged: _search,
            ),
            const SizedBox(height: 16),
            _loading
                ? const Expanded(
                  child: Center(child: CircularProgressIndicator()),
                )
                : Expanded(
                  child:
                      _items.isEmpty
                          ? const Center(child: Text('Aucun résultat'))
                          : AzListView(
                            data: _items,
                            itemCount: _items.length,
                            itemBuilder: (context, index) {
                              final item = _items[index];
                              return Card(
                                margin: const EdgeInsets.symmetric(vertical: 4),
                                child: ListTile(
                                  title: Text(
                                    item.acronym.acronym,
                                    style: const TextStyle(
                                      fontWeight: FontWeight.bold,
                                    ),
                                  ),
                                  subtitle: Text(item.acronym.meaning),
                                  trailing:
                                      item.acronym.source.isEmpty
                                          ? null
                                          : Text(item.acronym.source),
                                ),
                              );
                            },
                            itemScrollController: _itemScrollController,
                            itemPositionsListener: _itemPositionsListener,
                            indexBarData: _indexBarData,
                            indexBarWidth: 32,
                            indexBarItemHeight: 18,
                            indexBarMargin: const EdgeInsets.only(right: 4),
                            indexBarOptions: _buildIndexBarOptions(theme),
                            indexHintBuilder:
                                (context, tag) => _IndexHint(tag: tag),
                          ),
                ),
          ],
        ),
      ),
    );
  }

  IndexBarOptions _buildIndexBarOptions(ThemeData theme) {
    final colorScheme = theme.colorScheme;
    final textColor = colorScheme.onSurfaceVariant;
    return IndexBarOptions(
      needRebuild: true,
      downItemDecoration: BoxDecoration(
        color: colorScheme.primaryContainer.withValues(alpha: 0.35),
        shape: BoxShape.circle,
      ),
      downTextStyle: TextStyle(
        color: colorScheme.onPrimaryContainer,
        fontWeight: FontWeight.bold,
      ),
      selectItemDecoration: BoxDecoration(
        color: colorScheme.primaryContainer,
        shape: BoxShape.circle,
      ),
      selectTextStyle: TextStyle(
        color: colorScheme.onPrimaryContainer,
        fontWeight: FontWeight.bold,
      ),
      textStyle: TextStyle(
        color: textColor,
        fontSize: 12,
        fontWeight: FontWeight.w600,
      ),
      indexHintAlignment: Alignment.centerRight,
      indexHintOffset: const Offset(24, 0),
      indexHintWidth: 64,
      indexHintHeight: 64,
      indexHintDecoration: BoxDecoration(
        color: colorScheme.primary,
        shape: BoxShape.circle,
        boxShadow: [
          BoxShadow(
            color: colorScheme.shadow.withValues(alpha: 0.25),
            blurRadius: 12,
            offset: const Offset(0, 4),
          ),
        ],
      ),
      indexHintTextStyle:
          theme.textTheme.headlineSmall?.copyWith(
            color: colorScheme.onPrimary,
            fontWeight: FontWeight.bold,
          ) ??
          TextStyle(
            color: colorScheme.onPrimary,
            fontWeight: FontWeight.bold,
            fontSize: 24,
          ),
    );
  }
}

class _GlossaryItem extends ISuspensionBean {
  _GlossaryItem({required this.acronym, required this.tag});

  final Acronym acronym;
  final String tag;

  @override
  String getSuspensionTag() => tag;
}

class _IndexHint extends StatelessWidget {
  const _IndexHint({required this.tag});

  final String tag;

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    return Container(
      width: 64,
      height: 64,
      alignment: Alignment.center,
      decoration: BoxDecoration(
        color: theme.colorScheme.primary,
        shape: BoxShape.circle,
        boxShadow: [
          BoxShadow(
            color: theme.colorScheme.shadow.withValues(alpha: 0.25),
            blurRadius: 12,
            offset: const Offset(0, 4),
          ),
        ],
      ),
      child: Text(
        tag,
        style:
            theme.textTheme.headlineSmall?.copyWith(
              color: theme.colorScheme.onPrimary,
              fontWeight: FontWeight.bold,
            ) ??
            TextStyle(
              color: theme.colorScheme.onPrimary,
              fontWeight: FontWeight.bold,
              fontSize: 24,
            ),
      ),
    );
  }
}
