// lib/pages/download_manager_page.dart

import 'dart:async';
import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:intl/intl.dart';
import 'package:intl/date_symbol_data_local.dart';
import 'package:natinfo_flutter/features/downloads/data/download_service.dart';
import 'package:natinfo_flutter/features/pr_locator/data/csv_cache_service.dart';
import 'package:natinfo_flutter/shared/data_sources/dataset_ids.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:natinfo_flutter/shared/data_sources/source_override_store.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/update_service.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:natinfo_flutter/shared/utils/mounted_set_state_mixin.dart';
import 'package:natinfo_flutter/shared/utils/url_validator.dart';

enum DownloadItemKind {
  prNational,
  prDepartemental,
  natinfCategories,
  natinfInfractions,
}

class DownloadItem {
  final String label;
  final DownloadItemKind kind;
  final String id;
  final String? datasetId;
  final bool supportsCustomSource;

  const DownloadItem({
    required this.id,
    required this.label,
    required this.kind,
    this.datasetId,
    this.supportsCustomSource = false,
  });
}

class DownloadInfo {
  const DownloadInfo({required this.isDownloaded, required this.description});

  final bool isDownloaded;
  final String description;
}

class DownloadManagerPage extends StatefulWidget {
  const DownloadManagerPage({super.key});

  @override
  _DownloadManagerPageState createState() => _DownloadManagerPageState();
}

class _DownloadManagerPageState extends State<DownloadManagerPage>
    with MountedSetStateMixin<DownloadManagerPage> {
  final List<DownloadItem> _items = const [
    DownloadItem(
      id: DatasetIds.prNationalAutoroutes,
      label: 'Points de repère – National & Autoroutes',
      kind: DownloadItemKind.prNational,
      datasetId: DatasetIds.prNationalAutoroutes,
      supportsCustomSource: true,
    ),
    DownloadItem(
      id: DatasetIds.prDepartemental,
      label: 'Points de repère – Départemental',
      kind: DownloadItemKind.prDepartemental,
      datasetId: DatasetIds.prDepartemental,
      supportsCustomSource: true,
    ),
    DownloadItem(
      id: DatasetIds.natinfCategories,
      label: 'Catégories NATINFo',
      kind: DownloadItemKind.natinfCategories,
      datasetId: DatasetIds.natinfCategories,
      supportsCustomSource: true,
    ),
    DownloadItem(
      id: DatasetIds.natinf,
      label: 'Infractions NATINFo',
      kind: DownloadItemKind.natinfInfractions,
      supportsCustomSource: true,
      datasetId: DatasetIds.natinf,
    ),
  ];

  final CsvCacheService _csvCache = CsvCacheService();
  final SourceLogger _logger = SourceLogger(tag: 'Downloads');
  final Map<String, double> _progress = {};
  final Map<String, String> _status = {};
  Map<String, String> _customUrls = {};
  final Map<String, DownloadInfo> _itemInfos = {};
  Future<void>? _dateFormattingInit;
  late final Future<void> _initialUiReady;
  static const String _infoPrefsKeyPrefix = 'download_info_';
  Map<String, Uri> _defaultSourceUris = {};
  Map<String, bool> _datasetEditable = {};
  Map<String, String> _datasetNames = {};

  @override
  void initState() {
    super.initState();
    _logger.info('Opening DownloadManagerPage');
    _initialUiReady = _initializeUi();
    MatomoService().trackPage(
      title: 'Gestion des données téléchargées',
      path: '/downloads/',
    );
  }

  Future<void> _initializeUi() async {
    await _loadDefaultUris();
    await Future.wait([_restoreCachedInfos(), _loadCustomUrls()]);
    await _refreshAllItemInfos();
  }

  Future<void> _loadDefaultUris() async {
    final registry = await loadSourceRegistry();
    final defaults = <String, Uri>{};
    final editable = <String, bool>{};
    _datasetNames = registry.datasetNames;
    for (final item in _items.where((i) => i.supportsCustomSource)) {
      final dataset = item.datasetId ?? item.id;
      final sources = registry.sourcesForDataset(dataset);
      if (sources.isNotEmpty) {
        defaults[dataset] = sources.first.uri;
        editable[dataset] = sources.any((s) => s.userEditable);
      }
    }
    _defaultSourceUris = defaults;
    _datasetEditable = editable;
  }

  String _labelFor(DownloadItem item) {
    final dataset = item.datasetId ?? item.id;
    return _datasetNames[dataset] ?? item.label;
  }

  String _formatMetadata(SourceCacheMetadata meta) {
    final parts = <String>[];
    if (meta.sizeBytes != null) {
      parts.add('${_bytesToMb(meta.sizeBytes!)} Mo');
    }
    if (meta.lastUpdated != null) {
      parts.add('MAJ ${DateFormat.yMd().format(meta.lastUpdated!)}');
    }
    if (meta.sourceId != null) {
      parts.add('Source ${meta.sourceId}');
    }
    return parts.isEmpty ? 'Téléchargé' : 'Téléchargé (${parts.join(', ')})';
  }

  String _bytesToMb(int bytes) => (bytes / (1024 * 1024)).toStringAsFixed(1);

  Future<void> _restoreCachedInfos() async {
    final prefs = await SharedPreferences.getInstance();
    final restored = <String, DownloadInfo>{};
    for (final item in _items) {
      final raw = prefs.getString('$_infoPrefsKeyPrefix${item.id}');
      if (raw == null) continue;
      final sep = raw.indexOf('|');
      if (sep == -1) continue;
      final downloadedFlag = raw.substring(0, sep);
      final description = raw.substring(sep + 1);
      restored[item.id] = DownloadInfo(
        isDownloaded: downloadedFlag == '1',
        description: description,
      );
    }
    if (restored.isNotEmpty) {
      safeSetState(() {
        _itemInfos.addAll(restored);
      });
    }
  }

  Future<void> _persistItemInfo(String itemId, DownloadInfo info) async {
    final prefs = await SharedPreferences.getInstance();
    final raw = '${info.isDownloaded ? '1' : '0'}|${info.description}';
    await prefs.setString('$_infoPrefsKeyPrefix$itemId', raw);
  }

  Future<void> _loadCustomUrls() async {
    final prefs = await SharedPreferences.getInstance();
    final Map<String, String> urls = {};
    for (final item in _items.where((i) => i.supportsCustomSource)) {
      final dataset = item.datasetId ?? item.id;
      final defaultUrl =
          _defaultSourceUris[dataset]?.toString() ??
          NatDataService.defaultApiBase;
      final saved = prefs.getString(SourceOverrideStore.keyFor(dataset));
      urls[item.id] = (saved ?? defaultUrl).trim();
    }
    safeSetState(() {
      _customUrls = urls;
    });
    if (urls.isNotEmpty) {
      final entries = urls.entries.map((e) => '${e.key}=${e.value}').join(', ');
      _logger.info('Loaded custom source URLs: $entries');
    }
  }

  Future<bool> _saveCustomUrl(DownloadItem item, String url) async {
    final trimmed = url.trim();
    if (!UrlValidator.isSecureHttpUrl(trimmed)) {
      _logger.warn('Rejected custom URL for ${item.id}: $trimmed');
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Veuillez saisir une URL HTTPS valide.')),
      );
      return false;
    }

    final prefs = await SharedPreferences.getInstance();
    final dataset = item.datasetId ?? item.id;
    await prefs.setString(SourceOverrideStore.keyFor(dataset), trimmed);
    safeSetState(() {
      _customUrls[item.id] = trimmed;
    });
    _logger.info('Custom URL saved for ${item.id}: $trimmed');
    return true;
  }

  Future<DownloadInfo> _getItemInfo(DownloadItem item) async {
    _logger.info('Refreshing download info for ${item.id}');
    switch (item.kind) {
      case DownloadItemKind.prNational:
      case DownloadItemKind.prDepartemental:
        final datasetId = item.datasetId!;
        final hasData = await _csvCache.hasDataset(datasetId);
        if (!hasData) {
          _logger.warn('Dataset $datasetId missing on device');
          return const DownloadInfo(
            isDownloaded: false,
            description: 'Non téléchargé',
          );
        }
        final meta = await _csvCache.getDatasetMetadata(datasetId);
        if (meta != null) {
          final source = meta.sourceId ?? 'unknown';
          final updated = meta.lastUpdated?.toIso8601String() ?? 'unknown';
          _logger.info(
            'Dataset $datasetId metadata found (source: $source, updated: $updated)',
          );
        }
        return DownloadInfo(
          isDownloaded: true,
          description: meta != null ? _formatMetadata(meta) : 'Téléchargé',
        );
      case DownloadItemKind.natinfCategories:
        final status = await NatDataService.getCategoriesStatus();
        _logger.info(
          'NAT categories status: ${status.description} (hasData: ${status.hasData})',
        );
        return DownloadInfo(
          isDownloaded: status.hasData,
          description: status.description,
        );
      case DownloadItemKind.natinfInfractions:
        final status = await NatDataService.getNatinfStatus();
        _logger.info(
          'NAT infractions status: ${status.description} (hasData: ${status.hasData})',
        );
        return DownloadInfo(
          isDownloaded: status.hasData,
          description: status.description,
        );
    }
  }

  Future<void> _ensureDateFormattingInitialized() {
    return _dateFormattingInit ??= initializeDateFormatting(Intl.defaultLocale);
  }

  Future<void> _refreshItemInfo(DownloadItem item) async {
    await _ensureDateFormattingInitialized();
    final info = await _getItemInfo(item);
    safeSetState(() {
      _itemInfos[item.id] = info;
    });
    await _persistItemInfo(item.id, info);
  }

  Future<void> _refreshAllItemInfos() {
    return Future.wait(_items.map(_refreshItemInfo));
  }

  Future<void> _downloadItem(DownloadItem item) async {
    _logger.info('Download requested for ${item.id} (${item.kind})');
    safeSetState(() {
      _status[item.id] = 'Téléchargement...';
    });
    try {
      switch (item.kind) {
        case DownloadItemKind.prNational:
        case DownloadItemKind.prDepartemental:
          await _csvCache.fetchDataset(item.datasetId!, force: true);
          safeSetState(() {
            _status[item.id] = 'Téléchargement terminé';
          });
          _logger.info('Dataset ${item.datasetId} download completed');
          break;
        case DownloadItemKind.natinfCategories:
          await NatDataService.downloadNatCategories(
            context: context,
            onStatus:
                (s) => safeSetState(() {
                  _status[item.id] = s;
                }),
          );
          _logger.info('NAT categories download completed');
          break;
        case DownloadItemKind.natinfInfractions:
          await NatDataService.downloadNatData(
            context: context,
            onProgress:
                (p) => safeSetState(() {
                  _progress[item.id] = p;
                }),
            onStatus:
                (s) => safeSetState(() {
                  _status[item.id] = s;
                }),
          );
          _logger.info('NAT infractions download completed');
          break;
      }
      unawaited(_refreshItemInfo(item));
    } catch (e, stack) {
      _logger.warn('Download failed for ${item.id}: $e\n$stack');
      safeSetState(() {
        _status[item.id] = 'Erreur : ${e.toString()}';
        _progress.remove(item.id);
      });
    }
  }

  Future<void> _deleteItem(DownloadItem item) async {
    _logger.info('Delete requested for ${item.id}');
    try {
      switch (item.kind) {
        case DownloadItemKind.prNational:
        case DownloadItemKind.prDepartemental:
          await _csvCache.deleteDataset(item.datasetId!);
          break;
        case DownloadItemKind.natinfCategories:
          await NatDataService.deleteNatCategories();
          break;
        case DownloadItemKind.natinfInfractions:
          await NatDataService.deleteNatData();
          break;
      }
      safeSetState(() {
        _status[item.id] = 'Supprimé';
        _progress.remove(item.id);
      });
      _logger.info('Deletion completed for ${item.id}');
      unawaited(_refreshItemInfo(item));
    } catch (e, stack) {
      _logger.warn('Deletion failed for ${item.id}: $e\n$stack');
      safeSetState(() {
        _status[item.id] = 'Erreur : ${e.toString()}';
      });
    }
  }

  void _editSource(DownloadItem item) {
    final dataset = item.datasetId ?? item.id;
    final editable = _datasetEditable[dataset] ?? true;
    if (!item.supportsCustomSource || !editable) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(
          content: Text(
            'La source de ce jeu de données ne peut pas être modifiée.',
          ),
        ),
      );
      return;
    }
    final ctrl = TextEditingController(
      text:
          _customUrls[item.id] ??
          _defaultSourceUris[item.datasetId ?? item.id]?.toString() ??
          NatDataService.defaultApiBase,
    );
    showDialog<void>(
      context: context,
      builder:
          (ctx) => AlertDialog(
            title: Text('Source – ${_labelFor(item)}'),
            content: TextField(
              controller: ctrl,
              decoration: const InputDecoration(hintText: 'Nouvelle URL'),
            ),
            actions: [
              IconButton(
                icon: const Icon(Icons.close),
                onPressed: () => Navigator.pop(ctx),
              ),
              IconButton(
                icon: const Icon(Icons.check),
                onPressed: () async {
                  final saved = await _saveCustomUrl(item, ctrl.text);
                  if (!saved) return;
                  if (context.mounted) {
                    Navigator.pop(ctx);
                  }
                },
              ),
            ],
          ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(title: const Text('Gestion des données téléchargées')),
      body: FutureBuilder<void>(
        future: _initialUiReady,
        builder: (context, snapshot) {
          if (snapshot.connectionState != ConnectionState.done) {
            return const Center(child: CircularProgressIndicator());
          }
          return ListView.builder(
            itemCount: _items.length,
            itemBuilder: (_, i) {
              final item = _items[i];
              final info = _itemInfos[item.id];
              final downloaded = info?.isDownloaded ?? false;
              final dataset = item.datasetId ?? item.id;
              final canEdit =
                  item.supportsCustomSource &&
                  (_datasetEditable[dataset] ?? true);

              return Card(
                margin: const EdgeInsets.symmetric(horizontal: 16, vertical: 8),
                shape: RoundedRectangleBorder(
                  borderRadius: BorderRadius.circular(12),
                ),
                child: Padding(
                  padding: const EdgeInsets.all(12),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      ListTile(
                        contentPadding: EdgeInsets.zero,
                        title: Text(_labelFor(item)),
                        subtitle: Text(
                          info?.description ?? 'Chargement en cours…',
                        ),
                        trailing:
                            info == null
                                ? const SizedBox(
                                  width: 24,
                                  height: 24,
                                  child: CircularProgressIndicator(
                                    strokeWidth: 2,
                                  ),
                                )
                                : Row(
                                  mainAxisSize: MainAxisSize.min,
                                  children: [
                                    IconButton(
                                      icon: Icon(
                                        downloaded
                                            ? Icons.refresh
                                            : Icons.download,
                                      ),
                                      onPressed: () => _downloadItem(item),
                                    ),
                                    if (downloaded)
                                      IconButton(
                                        icon: const Icon(Icons.delete),
                                        color: Colors.redAccent,
                                        onPressed: () => _deleteItem(item),
                                      ),
                                    if (canEdit)
                                      IconButton(
                                        icon: const Icon(Icons.edit),
                                        color: Colors.grey,
                                        onPressed: () => _editSource(item),
                                      ),
                                  ],
                                ),
                      ),
                      if (_progress[item.id] != null &&
                          _progress[item.id]! < 1.0)
                        Padding(
                          padding: const EdgeInsets.only(top: 8),
                          child: LinearProgressIndicator(
                            value: _progress[item.id],
                          ),
                        ),
                      if (_status[item.id] != null)
                        Padding(
                          padding: const EdgeInsets.only(top: 8),
                          child: Text(
                            _status[item.id]!,
                            style: Theme.of(context).textTheme.bodySmall,
                          ),
                        ),
                    ],
                  ),
                ),
              );
            },
          );
        },
      ),
    );
  }
}
