import 'dart:convert';
import 'dart:io';
import 'dart:math';

import 'package:device_info_plus/device_info_plus.dart';
import 'package:flutter/widgets.dart';
import 'package:http/http.dart' as http;
import 'package:package_info_plus/package_info_plus.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'package:natinfo_flutter/app/config/app_config.dart';

class MatomoService {
  static final MatomoService _instance = MatomoService._internal();
  factory MatomoService() => _instance;
  MatomoService._internal();

  // ——— à adapter si besoin ———
  static const _endpoint = 'https://natinfo.app/matomo.php';
  static const _siteId = '8';
  static const _baseSiteUrl = 'https://natinfo.app';

  // IDs de Custom Dimensions (Administration > Mesurables > Custom Dimensions)
  // Visit-scope (portée "Visite") :
  static const _cdChannel = 1; // "channel": app / web
  static const _cdAppVersion = 2; // "app_version": 1.2.3
  static const _cdDeviceModel = 3; // "device_model": SM-G781B / iPhone15,4 …
  static const _cdOsName = 4; // "os_name": Android / iOS
  static const _cdOsVersion = 5; // "os_version": 13 / 17.5 …
  static const _cdBuildNumber = 6; // "build": 120

  String? _visitorId;

  Future<void> _ensureVisitorId() async {
    if (_visitorId != null) return;
    final prefs = await SharedPreferences.getInstance();
    _visitorId = prefs.getString('matomoVisitorId');
    if (_visitorId == null) {
      final r = Random.secure();
      final bytes = List<int>.generate(16, (_) => r.nextInt(256));
      final hex = bytes.map((b) => b.toRadixString(16).padLeft(2, '0')).join();
      _visitorId = hex;
      await prefs.setString('matomoVisitorId', hex);
    }
  }

  Future<bool> _canTrack() async {
    final prefs = await SharedPreferences.getInstance();
    final enabled = prefs.getBool('matomoTrackingEnabled') ?? true;
    return enabled && !AppConfig.forceOffline;
  }

  /// Construit un UA "comme un navigateur", bien reconnu par Matomo/DeviceDetector.
  Future<String> _buildUserAgent() async {
    final pkg = await PackageInfo.fromPlatform();
    final di = DeviceInfoPlugin();

    if (Platform.isAndroid) {
      final a = await di.androidInfo;
      final os = a.version.release ?? Platform.operatingSystemVersion;
      final model = a.model ?? 'Android';
      // UA mobile chromé générique + suffixe Natinfo
      return 'Mozilla/5.0 (Linux; Android $os; $model) '
          'AppleWebKit/537.36 (KHTML, like Gecko) '
          'Mobile Safari/537.36 Natinfo/${pkg.version}';
    } else if (Platform.isIOS) {
      final i = await di.iosInfo;
      final sys = i.systemVersion ?? Platform.operatingSystemVersion; // "17.5"
      final major = (sys.split('.').first);
      final machine = i.utsname.machine; // ex. "iPhone15,4"
      final deviceClass =
          (i.model?.toLowerCase().contains('ipad') ?? false)
              ? 'iPad'
              : 'iPhone';
      return 'Mozilla/5.0 ($deviceClass; CPU $deviceClass OS ${sys.replaceAll('.', '_')} like Mac OS X) '
          'AppleWebKit/605.1.15 (KHTML, like Gecko) '
          'Version/$major.0 Mobile/15E148 Safari/604.1 '
          'Natinfo/${pkg.version} ($machine)';
    }
    // Fallback desktop (rare en prod pour une app)
    return 'Natinfo/${pkg.version} (${Platform.operatingSystem} ${Platform.operatingSystemVersion})';
  }

  Future<Map<String, String>> _buildCommonParams() async {
    await _ensureVisitorId();

    final pkg = await PackageInfo.fromPlatform();
    final di = DeviceInfoPlugin();

    String osName =
        Platform.isAndroid
            ? 'Android'
            : Platform.isIOS
            ? 'iOS'
            : Platform.operatingSystem;
    String osVersion;
    String deviceModel;

    if (Platform.isAndroid) {
      final a = await di.androidInfo;
      osVersion = a.version.release ?? Platform.operatingSystemVersion;
      deviceModel = '${a.brand ?? ''} ${a.model ?? ''}'.trim();
    } else if (Platform.isIOS) {
      final i = await di.iosInfo;
      osVersion = i.systemVersion ?? Platform.operatingSystemVersion;
      deviceModel = i.utsname.machine; // iPhone15,4 …
    } else {
      osVersion = Platform.operatingSystemVersion;
      deviceModel = Platform.operatingSystem;
    }

    final locale = Platform.localeName; // ex. fr_FR
    final view = WidgetsBinding.instance.platformDispatcher.views.first;
    final width = (view.physicalSize.width / view.devicePixelRatio).round();
    final height = (view.physicalSize.height / view.devicePixelRatio).round();

    // Fallback (si Custom Dimensions non installées côté Matomo)
    final cvar = jsonEncode({
      '1': ['Channel', 'app'],
      '2': ['App Version', pkg.version],
      '3': ['Device', deviceModel],
      '4': ['OS', '$osName $osVersion'],
      '5': ['Build', pkg.buildNumber],
    });

    // NB: on met les dimensions *en plus* : si l’instance n’a pas le plugin,
    // Matomo ignore 'dimensionX' mais prendra _cvar.
    final params = <String, String>{
      'idsite': _siteId,
      'rec': '1',
      '_id': _visitorId!,
      'lang': locale,
      'res': '${width}x$height',
      'urlref': _baseSiteUrl,
      '_cvar': cvar,

      // Custom Dimensions (si activées)
      'dimension$_cdChannel': 'app',
      'dimension$_cdAppVersion': pkg.version,
      'dimension$_cdDeviceModel': deviceModel,
      'dimension$_cdOsName': osName,
      'dimension$_cdOsVersion': osVersion,
      'dimension$_cdBuildNumber': pkg.buildNumber,
    };

    // UA complet (même valeur que le header)
    params['ua'] = await _buildUserAgent();
    return params;
  }

  Future<void> _sendHit(Map<String, String> params) async {
    final uri = Uri.parse(_endpoint).replace(queryParameters: params);
    final ua = params['ua'] ?? await _buildUserAgent();

    try {
      await http.get(uri, headers: {'User-Agent': ua});
    } catch (_) {
      // ne jamais faire planter l’app pour l’analytics
    }
  }

  Future<void> trackPage({required String title, required String path}) async {
    if (!await _canTrack()) return;
    final p = await _buildCommonParams();
    p['action_name'] = title;
    // on garde l’URL du site pour agréger les rapports “contenu”
    p['url'] = '$_baseSiteUrl$path';
    await _sendHit(p);
  }

  Future<void> trackSearch({
    required String keyword,
    required int count,
  }) async {
    if (!await _canTrack()) return;
    final p = await _buildCommonParams();
    p['search'] = keyword;
    p['search_count'] = count.toString();
    p['url'] = '$_baseSiteUrl/search/?q=${Uri.encodeComponent(keyword)}';
    await _sendHit(p);
  }

  // Bonus: événements (clics, partages, etc.)
  Future<void> trackEvent({
    required String category,
    required String action,
    String? name,
    num? value,
  }) async {
    if (!await _canTrack()) return;
    final p = await _buildCommonParams();
    p['e_c'] = category;
    p['e_a'] = action;
    if (name != null) p['e_n'] = name;
    if (value != null) p['e_v'] = value.toString();
    // URL neutre pour les events :
    p['url'] = '$_baseSiteUrl/app-event';
    await _sendHit(p);
  }
}
