import 'dart:io';
import 'package:flutter/material.dart';
import 'package:latlong2/latlong.dart';
import 'package:natinfo_flutter/features/pr_locator/data/location_service.dart';
import 'package:natinfo_flutter/features/pr_locator/data/pr_repository.dart';
import 'package:natinfo_flutter/features/pr_locator/domain/entities/pr_point.dart';
import 'package:natinfo_flutter/shared/data_sources/dataset_ids.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry_loader.dart';

enum LoadState { loading, missing, ready, error }

class PrLocatorViewModel extends ChangeNotifier {
  final PrRepository _repo;
  final LocationService _locService;
  final Future<Map<String, String>> Function() _datasetNamesLoader;

  /// Current loading state of the view model.
  LoadState state = LoadState.loading;

  /// Last error message surfaced to the UI.
  String? errorMessage;

  /// Progress or status message while loading datasets.
  String loadingMessage = 'Données non téléchargées';

  /// Loaded datasets keyed by dataset id.
  Map<String, List<PrPoint>> datasets = {};

  /// Human readable names keyed by dataset id.
  Map<String, String> datasetNames = {};

  /// Currently selected dataset id.
  String selectedDataset = DatasetIds.prNationalAutoroutes;

  /// Points displayed on the map: user tap then nearest PR.
  List<LatLng> markers = [];

  /// Human readable result for the last lookup.
  String? result;

  PrLocatorViewModel(
    this._repo,
    this._locService, {
    Future<Map<String, String>> Function()? datasetNamesLoader,
  }) : _datasetNamesLoader = datasetNamesLoader ?? _defaultDatasetNamesLoader {
    _initialize();
  }

  Future<void> _initialize() async {
    state = LoadState.loading;
    loadingMessage = 'Vérification des données…';
    notifyListeners();
    await _loadDatasetNames();

    if (await _repo.isDownloaded) {
      if (!_repo.hasCache) {
        await _repo.cacheData();
      }
      await loadData();
      return;
    }

    state = LoadState.missing;
    loadingMessage = 'Données non téléchargées';
    notifyListeners();
  }

  Future<void> _loadDatasetNames() async {
    try {
      datasetNames = await _datasetNamesLoader();
    } catch (_) {
      datasetNames = {};
    }
    notifyListeners();
  }

  static Future<Map<String, String>> _defaultDatasetNamesLoader() async {
    final registry = await loadSourceRegistry();
    return registry.datasetNames;
  }

  /// Whether "locate me" is supported on this platform.
  bool get isLocateMeAvailable => !Platform.isLinux;

  /// Loads PR datasets from cache or network.
  Future<void> loadData() async {
    if (_repo.hasCache) {
      datasets = await _repo.loadAll();
      _selectDefaultDatasetIfNeeded();
      state = LoadState.ready;
      notifyListeners();
      return;
    }

    state = LoadState.loading;
    loadingMessage = 'Initialisation du téléchargement…';
    notifyListeners();
    try {
      progressCallback(String progress) {
        loadingMessage = progress;
        notifyListeners();
      }

      datasets = await _repo.loadAll(progressCallback);
      _selectDefaultDatasetIfNeeded();
      state = LoadState.ready;
    } catch (e) {
      errorMessage = e.toString();
      state = LoadState.error;
    }
    notifyListeners();
  }

  /// Selects the dataset used for searches and clears existing results.
  void selectDataset(String datasetId) {
    if (selectedDataset == datasetId) return;
    selectedDataset = datasetId;
    _clearResults();
  }

  /// Human-friendly label for a dataset id.
  String datasetLabel(String datasetId) {
    return datasetNames[datasetId] ?? datasetId;
  }

  void _selectDefaultDatasetIfNeeded() {
    if (datasets.isEmpty) return;
    if (!datasets.containsKey(selectedDataset)) {
      selectedDataset = datasets.keys.first;
    }
  }

  /// Clears markers and the last computed result.
  void resetMap() {
    _clearResults();
  }

  void _clearResults() {
    markers.clear();
    result = null;
    errorMessage = null;
    notifyListeners();
  }

  /// Handles map taps by locating the nearest PR in the selected dataset.
  void handleTap(LatLng pt) {
    const tolerance = 5.0; // mètres
    final distCalc = const Distance();

    if (markers.isNotEmpty) {
      bool already = markers.any(
        (m) => distCalc.as(LengthUnit.Meter, pt, m) < tolerance,
      );
      if (already) {
        notifyListeners();
        return;
      }
    }

    try {
      final list = datasets[selectedDataset]!;
      if (list.isEmpty) {
        errorMessage = 'Aucun point PR dans le dataset sélectionné';
        notifyListeners();
        return;
      }

      final nearest = list.reduce((a, b) {
        final da = distCalc.as(LengthUnit.Meter, pt, LatLng(a.lat, a.lon));
        final db = distCalc.as(LengthUnit.Meter, pt, LatLng(b.lat, b.lon));
        return da < db ? a : b;
      });

      final dist =
          distCalc
              .as(LengthUnit.Meter, pt, LatLng(nearest.lat, nearest.lon))
              .round();
      markers = [pt, LatLng(nearest.lat, nearest.lon)];
      result = 'PR ${nearest.pr} (${nearest.route}) — à $dist m';
    } catch (e) {
      errorMessage = 'Erreur lors du calcul du point le plus proche';
    }
    notifyListeners();
  }

  /// Attempts to locate the user via GPS and find the nearest PR.
  Future<void> locateMe() async {
    if (!isLocateMeAvailable) {
      errorMessage = 'Localisation non disponible sur Linux.';
      notifyListeners();
      return;
    }
    try {
      await _locService.ensurePermission();
      final pt = await _locService.getCurrentLocation();
      handleTap(pt);
    } catch (e) {
      errorMessage = e.toString();
      notifyListeners();
    }
  }

  /// Geocodes an address and finds the nearest PR to that location.
  Future<void> locateAddress(String address) async {
    try {
      final pt = await _locService.geocode(address);
      handleTap(pt);
    } catch (e) {
      errorMessage = e.toString();
      notifyListeners();
    }
  }
}
